﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class AddRoleCommandHandler(
        IRoleRepository _roleRepository,
        IUnitOfWork<QMSDbContext> _uow,
        IMapper _mapper,
        IRoleClaimRepository _roleClaimRepository) : IRequestHandler<AddRoleCommand, RoleDto>
    {
        public async Task<RoleDto> Handle(AddRoleCommand request, CancellationToken cancellationToken)
        {
            var entityExist = await _roleRepository
                .FindByInclude(v => v.Name == request.Name, c => c.RoleClaims)
                .IgnoreQueryFilters()
                .FirstOrDefaultAsync();

            if (entityExist != null && !entityExist.IsDeleted)
            {
                var errorDto = new RoleDto
                {
                    StatusCode = 409,
                    Messages = new List<string> { "Role Name already exist." }
                };
                return errorDto;
            }

            // Update Role
            if (entityExist != null)
            {
                entityExist.Name = request.Name;
                entityExist.NormalizedName = request.Name;
                entityExist.IsDeleted = false;
                _roleRepository.Update(entityExist);

                // update Role Claim
                var roleClaimsToDelete = entityExist.RoleClaims.ToList();
                _roleClaimRepository.RemoveRange(roleClaimsToDelete);

                request.RoleClaims.ForEach(claim =>
                {
                    claim.ClaimType = claim.ClaimType.Replace(" ", "_");
                    claim.RoleId = entityExist.Id;
                });
                _roleClaimRepository.AddRange(_mapper.Map<List<RoleClaim>>(request.RoleClaims));
            }
            else
            {
                // add new role.
                entityExist = _mapper.Map<Role>(request);
                entityExist.Id = Guid.NewGuid();
                entityExist.NormalizedName = entityExist.Name;
                entityExist.RoleClaims.ToList().ForEach(claim => claim.ClaimType = claim.ClaimType.Replace(" ", "_"));
                _roleRepository.Add(entityExist);
            }

            if (await _uow.SaveAsync() <= -1)
            {
                var errorDto = new RoleDto
                {
                    StatusCode = 500,
                    Messages = new List<string> { "An unexpected fault happened. Try again later." }
                };
                return errorDto;
            }
            var entityDto = _mapper.Map<RoleDto>(entityExist);
            return entityDto;
        }
    }
}
