﻿using AutoMapper;
using MediatR;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class AddStorageSettingCommandHandler(
         IStorageSettingRepository _storageSettingRepository,
         IMapper _mapper,
          IUnitOfWork<QMSDbContext> _uow,
         StorageServiceFactory _storeageServiceFactory,
         PathHelper _pathHelper,
         IWebHostEnvironment _webHostEnvironment) : IRequestHandler<AddStorageSettingCommand, ServiceResponse<StorageSettingDto<string>>>
    {

        public async Task<ServiceResponse<StorageSettingDto<string>>> Handle(AddStorageSettingCommand request, CancellationToken cancellationToken)
        {
            var entityExist = await _storageSettingRepository.FindBy(c => c.Name == request.Name).FirstOrDefaultAsync();
            if (entityExist != null)
            {
                return ServiceResponse<StorageSettingDto<string>>.Return409("Storage Setting Name is already exists.");
            }

            string storagePath = Path.Combine(_webHostEnvironment.ContentRootPath, _pathHelper.DocumentPath, _pathHelper.TestFile);
            IsDummyFileCreatedIfNotExist(storagePath);
            IFormFile file = FileHelper.ConvertToIFormFile(storagePath);
            var storageService = _storeageServiceFactory.GetStorageService(request.StorageType);
            var storageSetting = new StorageSetting
            {
                Name = request.Name,
                StorageType = request.StorageType,
                IsDefault = request.IsDefault,
                JsonValue = request.JsonValue,
                EnableEncryption = request.EnableEncryption

            };
            var keyValutFile = await storageService.TestUploadFileAsync(file, storageSetting, "");
            if (string.IsNullOrEmpty(keyValutFile.FileName))
            {
                return ServiceResponse<StorageSettingDto<string>>.Return422("Storage Setting is not properly setup.");
            }
            if (request.IsDefault)
            {
                var isDefaultEntity = _storageSettingRepository.All.Where(c => c.IsDefault == true).FirstOrDefault();
                if (isDefaultEntity != null)
                {
                    isDefaultEntity.IsDefault = false;
                    _storageSettingRepository.Update(isDefaultEntity);
                }
            }
            var entity = _mapper.Map<StorageSetting>(request);
            entity.Id = Guid.NewGuid();
            _storageSettingRepository.Add(entity);
            if (await _uow.SaveAsync() <= 0)
            {
                return ServiceResponse<StorageSettingDto<string>>.Return500();
            }
            return ServiceResponse<StorageSettingDto<string>>.ReturnResultWith201(_mapper.Map<StorageSettingDto<string>>(entity));
        }

        private bool IsDummyFileCreatedIfNotExist(string path)
        {
            if (File.Exists(path))
            {
                return true;
            }
            else
            {
                string dummyText = "This is a dummy text file.\n" +
                           "It contains some sample text for demonstration purposes.\n" +
                           "Feel free to modify this content as needed.\n\n" +
                           "Here are some bullet points:\n" +
                           "- Point 1: Dummy text example.\n" +
                           "- Point 2: More dummy text.\n" +
                           "- Point 3: Even more dummy text.";

                // Create the text file and write the dummy text into it
                try
                {
                    File.WriteAllText(path, dummyText);
                    return true;
                }
                catch (Exception ex)
                {
                    throw ex;
                }
            }
        }
    }
}
