﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class UpdateStorageSettingCommandHandler(
        IStorageSettingRepository _storageSettingRepository,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow) : IRequestHandler<UpdateStorageSettingCommand, ServiceResponse<StorageSettingDto<string>>>
    {
        public async Task<ServiceResponse<StorageSettingDto<string>>> Handle(UpdateStorageSettingCommand request, CancellationToken cancellationToken)
        {
            var nameExists = await _storageSettingRepository.FindBy(c => c.Name == request.Name && c.Id != request.Id).FirstOrDefaultAsync();
            if (nameExists != null)
            {
                return ServiceResponse<StorageSettingDto<string>>.Return409("Storage Setting Name is already exists.");
            }
            var existingEntity = await _storageSettingRepository.FindBy(c => c.Id == request.Id).FirstOrDefaultAsync();
            if (existingEntity == null)
            {
                return ServiceResponse<StorageSettingDto<string>>.Return404("No record found.");
            }

            if (request.IsDefault)
            {
                var isDefaultEntity = _storageSettingRepository.All.Where(c => c.IsDefault == true && c.Id != request.Id).FirstOrDefault();
                if (isDefaultEntity != null)
                {
                    isDefaultEntity.IsDefault = false;
                    _storageSettingRepository.Update(isDefaultEntity);
                }
            }
            else
            {
                if (existingEntity.IsDefault)
                {
                    var localEntity = _storageSettingRepository.All.Where(c => c.StorageType == Data.StorageType.LOCAL_STORAGE).FirstOrDefault();
                    if (localEntity != null && localEntity.Id == existingEntity.Id)
                    {
                        request.IsDefault = true;
                    }
                    else
                    {
                        localEntity.IsDefault = true;
                        _storageSettingRepository.Update(localEntity);
                    }
                }
            }

            // If no issues, proceed with further logic...
            var storageSetting = _mapper.Map(request, existingEntity);
            _storageSettingRepository.Update(storageSetting);
            if (await _uow.SaveAsync() <= -1)
            {
                return ServiceResponse<StorageSettingDto<string>>.Return500();
            }
            return ServiceResponse<StorageSettingDto<string>>.ReturnResultWith201(_mapper.Map<StorageSettingDto<string>>(existingEntity));
        }
    }
}
