﻿using System;
using System.Threading;
using System.Threading.Tasks;
using AutoMapper;
using MediatR;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;
public class UpdateSupplierCommandHandler(
    ISupplierRepository supplierRepository,
    IUnitOfWork<QMSDbContext> unitOfWork,
    ILogger<UpdateSupplierCommandHandler> logger,
    IMapper mapper) : IRequestHandler<UpdateSupplierCommand, ServiceResponse<SupplierDto>>
{
    public async Task<ServiceResponse<SupplierDto>> Handle(UpdateSupplierCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var supplier = await supplierRepository.FindAsync(request.Id);
            if (supplier == null)
            {
                return ServiceResponse<SupplierDto>.Return404("Supplier not found.");
            }

            supplier = mapper.Map(request, supplier);
            supplierRepository.Update(supplier);

            if (await unitOfWork.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<SupplierDto>.Return500("Error while updating supplier.");
            }

            return ServiceResponse<SupplierDto>.ReturnResultWith201(mapper.Map<SupplierDto>(supplier));

        }
        catch (Exception ex)
        {
            logger.LogError(ex, "Error while updating supplier.");
            return ServiceResponse<SupplierDto>.Return500("Error while updating supplier.");
        }
    }
}
