﻿using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;
public class AddSupplierAuditCommandHandler(
   ISupplierAuditRepository _supplierAuditRepository,
   IUserRepository userRepository,
   ISupplierRepository supplierRepository,
   IMapper _mapper,
   IUnitOfWork<QMSDbContext> _uow,
   ILogger<AddSupplierCommandHandler> _logger) : IRequestHandler<AddSupplierAuditCommand, ServiceResponse<SupplierAuditDto>>
{
    public async Task<ServiceResponse<SupplierAuditDto>> Handle(AddSupplierAuditCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var supplier = await supplierRepository.FindAsync(request.SupplierId);
            if (supplier == null)
            {
                return ServiceResponse<SupplierAuditDto>.Return404();
            }
            var entity = _mapper.Map<SupplierAudit>(request);
            _supplierAuditRepository.Add(entity);
            if (await _uow.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<SupplierAuditDto>.Return500();
            }
            var user = await userRepository.All.Where(c => c.Id == entity.AuditorId).FirstOrDefaultAsync();
            if (user != null)
            {
                entity.Auditor = user;
            }
            var entityDto = _mapper.Map<SupplierAuditDto>(entity);
            return ServiceResponse<SupplierAuditDto>.ReturnResultWith201(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Erorr while saving SupplierAudit");
            return ServiceResponse<SupplierAuditDto>.Return500("Error while Saving SupplierAudit");
        }
    }
}
