﻿using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;
public class AddSupplierCertificationCommandHandler(
    ISupplierCertificationRepository supplierCertificationRepository,
    IUnitOfWork<QMSDbContext> _uow,
    IMapper mapper,
    StorageServiceFactory _storeageServiceFactory,
    IStorageSettingRepository _storageSettingRepository,
    ILogger<AddSupplierCertificationCommandHandler> _logger) : IRequestHandler<AddSupplierCertificationCommand, ServiceResponse<SupplierCertificationDto>>
{
    public async Task<ServiceResponse<SupplierCertificationDto>> Handle(AddSupplierCertificationCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var existEntity = await supplierCertificationRepository.All
          .Where(c => c.CertificationName == request.CertificationName)
          .FirstOrDefaultAsync();
            if (existEntity != null)
            {
                return ServiceResponse<SupplierCertificationDto>.Return409("SupplierCertification with same name already exists.");
            }
            if (request.File == null)
            {
                return ServiceResponse<SupplierCertificationDto>.ReturnFailed(409, "Please select the file.");
            }

            if (!FileSignatureHelper.IsFileSignatureValid(request.File))
            {
                return ServiceResponse<SupplierCertificationDto>.ReturnFailed(409, "Invalid file signature.");
            }
            long fileSizeInBytes = request.File.Length;
            // Convert file size to kilobytes or megabytes if necessary
            double fileSizeInKB = fileSizeInBytes / 1024.0;
            var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(null);

            var storageService = _storeageServiceFactory.GetStorageService(storeageSetting.StorageType);

            var fileNameKeyValut = await storageService.UploadFileAsync(request.File, storeageSetting, null);

            if (string.IsNullOrEmpty(fileNameKeyValut.FileName))
            {
                return ServiceResponse<SupplierCertificationDto>.Return422("Settings are not properly setup.");
            }
            var entity = mapper.Map<SupplierCertification>(request);
            entity.FilePath = fileNameKeyValut.FileName;
            supplierCertificationRepository.Add(entity);
            if (await _uow.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<SupplierCertificationDto>.Return500();
            }
            var entityDto = mapper.Map<SupplierCertificationDto>(entity);
            return ServiceResponse<SupplierCertificationDto>.ReturnResultWith201(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Erorr while saving SupplierCertification");
            return ServiceResponse<SupplierCertificationDto>.Return500("Error while Saving SupplierCertification");
        }
    }
}
