﻿using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using MediatR;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Data.Dto;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;
public class DownloadSupplierCertificationCommandHandler(
          ISupplierCertificationRepository supplierCertificationRepository,
        IStorageSettingRepository _storageSettingRepository,
        StorageServiceFactory _storageServiceFactory) : IRequestHandler<DownloadSupplierCertificationCommand, ServiceResponse<SupplierCertificationDownload>>
{
    public async Task<ServiceResponse<SupplierCertificationDownload>> Handle(DownloadSupplierCertificationCommand request, CancellationToken cancellationToken)
    {
        var supplierCertification = await supplierCertificationRepository.All
            .Where(c => c.Id == request.SupplierCertificationId)
            .FirstOrDefaultAsync();

        if (supplierCertification == null)
        {
            return ServiceResponse<SupplierCertificationDownload>.ReturnFailed(404, "SupplierCertification is not found");
        }

        var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(null);

        if (storeageSetting == null)
        {
            return ServiceResponse<SupplierCertificationDownload>.ReturnFailed(404, "Storage setting not found");
        }
        var storageService = _storageServiceFactory.GetStorageService(storeageSetting.StorageType);
        var fileResult = await storageService.DownloadFileAsync(supplierCertification.FilePath, storeageSetting.JsonValue, null, null);
        if (string.IsNullOrWhiteSpace(fileResult.ErrorMessage))
        {
            var extension = supplierCertification.FilePath;
            if (!extension.StartsWith("."))
                extension = "." + extension;
            SupplierCertificationDownload supplierCertificationDownload = new SupplierCertificationDownload
            {
                Data = fileResult.FileBytes,
                ContentType = FileHelper.GetMimeTypeByExtension(extension),
                FileName = supplierCertification.FileName,
            };
            return ServiceResponse<SupplierCertificationDownload>.ReturnResultWith200(supplierCertificationDownload);
        }
        return ServiceResponse<SupplierCertificationDownload>.ReturnFailed(500, fileResult.ErrorMessage);
    }
}
