﻿using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;
public class AddSupplierPerformanceCommandHandler(
   ISupplierPerformanceRepository supplierPerformanceRepository,
   ISupplierRepository supplierRepository,
   IUserRepository userRepository,
   IMapper _mapper,
   IUnitOfWork<QMSDbContext> _uow,
   ILogger<AddSupplierPerformanceCommandHandler> _logger) : IRequestHandler<AddSupplierPerformanceCommand, ServiceResponse<SupplierPerformanceDto>>
{
    public async Task<ServiceResponse<SupplierPerformanceDto>> Handle(AddSupplierPerformanceCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var supplier = await supplierRepository.FindAsync(request.SupplierId);
            if (supplier == null)
            {
                return ServiceResponse<SupplierPerformanceDto>.Return404();
            }
            var entity = _mapper.Map<SupplierPerformance>(request);
            supplierPerformanceRepository.Add(entity);
            if (await _uow.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<SupplierPerformanceDto>.Return500();
            }
            var user = await userRepository.All.Where(c => c.Id == entity.EvaluatedById).FirstOrDefaultAsync();
            if (user != null)
            {
                entity.EvaluatedBy = user;
            }
            var entityDto = _mapper.Map<SupplierPerformanceDto>(entity);
            return ServiceResponse<SupplierPerformanceDto>.ReturnResultWith201(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Erorr while saving SupplierPerformance");
            return ServiceResponse<SupplierPerformanceDto>.Return500("Error while Saving SupplierPerformance");
        }
    }
}
