﻿using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;
public class AddSupplierQualificationCommandHandler(
   ISupplierQualificationRepository supplierQualificationRepository,
   ISupplierRepository supplierRepository,
   IUserRepository userRepository,
   IMapper _mapper,
   IUnitOfWork<QMSDbContext> _uow,
   ILogger<AddSupplierPerformanceCommandHandler> _logger) : IRequestHandler<AddSupplierQualificationCommand, ServiceResponse<SupplierQualificationDto>>
{
    public async Task<ServiceResponse<SupplierQualificationDto>> Handle(AddSupplierQualificationCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var supplier = await supplierRepository.FindAsync(request.SupplierId);
            if (supplier == null)
            {
                return ServiceResponse<SupplierQualificationDto>.Return404();
            }
            var entity = _mapper.Map<SupplierQualification>(request);
            supplierQualificationRepository.Add(entity);
            if (await _uow.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<SupplierQualificationDto>.Return500();
            }
            var user = await userRepository.All.Where(c => c.Id == entity.QualifiedById).FirstOrDefaultAsync();
            if (user != null)
            {
                entity.QualifiedBy = user;
            }
            var entityDto = _mapper.Map<SupplierQualificationDto>(entity);
            return ServiceResponse<SupplierQualificationDto>.ReturnResultWith201(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Erorr while saving SupplierQualification");
            return ServiceResponse<SupplierQualificationDto>.Return500("Error while Saving SupplierQualification");
        }
    }
}
