﻿using System.Threading;
using System.Threading.Tasks;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using MediatR;
using Microsoft.AspNetCore.Identity;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;

namespace QualityManagement.MediatR;

public class RegisterEmployeeUserCommandHandler(
    UserManager<User> userManager,
    ILogger<RegisterEmployeeUserCommandHandler> logger,
    IRoleRepository roleRepository,
    IUserRoleRepository userRoleRepository,
    IUnitOfWork<QMSDbContext> unitOfWork
    ) : IRequestHandler<RegisterEmployeeUserCommand, ServiceResponse<bool>>
{
    public async Task<ServiceResponse<bool>> Handle(RegisterEmployeeUserCommand request, CancellationToken cancellationToken)
    {
        var appUser = await userManager.FindByIdAsync(request.Id.ToString());
        if (appUser == null)
        {
            logger.LogError("Employee Does not exists.");
            return ServiceResponse<bool>.Return409("Error while registration.");
        }


        if (!string.IsNullOrEmpty(request.Password))
        {
            string code = await userManager.GeneratePasswordResetTokenAsync(appUser);
            IdentityResult passwordResult = await userManager.ResetPasswordAsync(appUser, code, request.Password);
            if (!passwordResult.Succeeded)
            {
                return ServiceResponse<bool>.Return500();
            }
        }

        // assign employee role to user.
        var role = await roleRepository.GetRoleByUserType(UserType.Employee);
        if (role != null)
        {
            var existingRole = await userRoleRepository.All.FirstOrDefaultAsync(d => d.UserId == appUser.Id && d.RoleId == role.Id);
            if (existingRole == null)
            {
                userRoleRepository.Add(new UserRole
                {
                    UserId = appUser.Id,
                    RoleId = role.Id,
                });
                await unitOfWork.SaveAsync();
            }
        }

        appUser.FirstName = request.FirstName;
        appUser.LastName = request.LastName;
        appUser.IsSignup = true;
        appUser.PhoneNumber = request.PhoneNumber;
        appUser.Address = request.Address;

        IdentityResult result = await userManager.UpdateAsync(appUser);
        if (!result.Succeeded)
        {
            return ServiceResponse<bool>.Return500();
        }

        return ServiceResponse<bool>.ReturnSuccess();
    }
}