﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Data.Dto;

namespace QualityManagement.Repository;
public class AuditList : List<AuditDto>
{
    public int Skip { get; private set; }
    public int TotalPages { get; private set; }
    public int PageSize { get; private set; }
    public int TotalCount { get; private set; }

    public AuditList(List<AuditDto> items, int count, int skip, int pageSize)
    {
        TotalCount = count;
        PageSize = pageSize;
        Skip = skip;
        TotalPages = (int)Math.Ceiling(count / (double)pageSize);
        AddRange(items);
    }
    public AuditList()
    {

    }

    public async Task<AuditList> Create(IQueryable<Data.Entities.Audit> source, int skip, int pageSize)
    {
        var count = await GetCount(source);
        var dtoList = await GetDtos(source, skip, pageSize);
        var dtoPageList = new AuditList(dtoList, count, skip, pageSize);
        return dtoPageList;
    }

    public async Task<int> GetCount(IQueryable<Data.Entities.Audit> source)
    {
        return await source.AsNoTracking().CountAsync();
    }

    public async Task<List<AuditDto>> GetDtos(IQueryable<Data.Entities.Audit> source, int skip, int pageSize)
    {
        try
        {
            var auditTemplates = await source
                .Skip(skip)
                .Take(pageSize)
                .AsNoTracking()
                .Select(c => new AuditDto
                {
                    Id = c.Id,
                    AuditNumber = c.AuditNumber,
                    AuditTemplateId = c.AuditTemplateId,
                    EffectiveScore = c.EffectiveScore,
                    TotalScore = c.TotalScore,
                    CreatedDate = c.CreatedDate,
                    DepartmentName = c.AuditTemplate.Department != null ? c.AuditTemplate.Department.Name : null,
                    AuditTemplate = new AuditTemplateDto
                    {
                        Id = c.AuditTemplate.Id,
                        Name = c.AuditTemplate.Name,
                        Description = c.AuditTemplate.Description,
                        AuditType = c.AuditTemplate.AuditType,
                        AuditId = c.AuditTemplate.Audits
                            .Where(a => a.Id == c.Id)
                            .Select(a => a.Id)
                            .FirstOrDefault(),
                        IconName = c.AuditTemplate.IconName,
                        DepartmentId = c.AuditTemplate.DepartmentId,
                        TotalQuestions = c.AuditTemplate.Questions.Count,
                        IsActive = c.AuditTemplate.IsActive
                    },
                    NonConformances = c.NonConformances.Select(c => new NonConformanceDto
                    {
                        Id = c.Id,
                        Title = c.Title,
                        DepartmentId = c.DepartmentId.Value,
                        Description = c.Description,
                        AuditId = c.AuditId,
                        ClosedDate = c.ClosedDate,
                        NCNumber = c.NCNumber,
                        ReportedById = c.ReportedById,
                        Severity = c.Severity,
                        Status = c.Status,
                        DepartmentName = c.Department != null ? c.Department.Name : null,
                        AuditName = c.Audit != null ? c.Audit.Title : null,
                        ReportedByName = c.ReportedBy != null ? $"{c.ReportedBy.FirstName}{c.ReportedBy.LastName}" : null,
                    }).ToList(),
                    CapaRequests = c.CapaRequests.Select(c => new CapaRequestDto
                    {
                        Id = c.Id,
                        ReferenceNo = c.ReferenceNo,
                        Title = c.Title,
                        Description = c.Description,
                        RaisedById = c.RaisedById,
                        RaisedByName = c.RaisedBY != null ? $"{c.RaisedBY.FirstName} {c.RaisedBY.LastName}" : null,
                        NonConformanceId = c.NonConformanceId.Value,
                        AuditId = c.AuditId,
                        AuditName = c.Audit != null ? c.Audit.Title : null,
                        CapaRequestStatus = c.CapaRequestStatus,
                        Priority = c.Priority,
                        DueDate = c.DueDate
                    }).ToList(),
                    AuditReviewers = c.AuditReviewers.OrderByDescending(d => d.ReviewedOn).Select(c => new AuditReviewerDto
                    {
                        Id = c.Id,
                        AuditId = c.AuditId,
                        Comments = c.Comments,
                        ReviewerId = c.ReviewerId,
                        ReviewedOn = c.ReviewedOn,
                    }).ToList(),
                    AuditReviewComments = c.AuditReviewComments.OrderByDescending(d => d.CreatedDate).Select(c => new AuditReviewCommentDto
                    {
                        Id = c.Id,
                        AuditId = c.AuditId,
                        Comments = c.Comments,
                        CreatedDate = c.CreatedDate
                    }).ToList(),
                    Status = c.Status,
                    Title = c.Title,
                    AuditorName = c.Auditor != null ? $"{c.Auditor.FirstName} {c.Auditor.LastName}" : null,
                    AuditorId = c.AuditorId
                })
                .ToListAsync();
            return auditTemplates;
        }
        catch (Exception ex)
        {
            throw new DataException("Error while getting Audit", ex);
        }
    }
}