﻿using Microsoft.EntityFrameworkCore;
using QualityManagement.Common.GenericRepository;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Data.Resources;
using QualityManagement.Domain;
using System;
using System.Linq;
using System.Threading.Tasks;

namespace QualityManagement.Repository;

public class AuditRepository(
    IPropertyMappingService _propertyMappingService,
    UserInfoToken userInfoToken,
    IUnitOfWork<QMSDbContext> uow) : GenericRepository<Audit, QMSDbContext>(uow), IAuditRepository
{


    public async Task<string> GenerateAuditNumberAsync()
    {
        int year = DateTime.UtcNow.Year;
        int nextNumber = await All.Where(d => d.AuditNumber.StartsWith($"AD-{year}-"))
            .CountAsync() + 1;
        return $"AD-{year}-{nextNumber:D4}"; // Example: AD-2024-00001
    }

    public async Task<AuditList> GetAudits(AuditResource auditResource)
    {

        var collectionBeforePaging = AllIncluding(c => c.AuditTemplate, c => c.Auditor, c => c.NonConformances, c => c.CapaRequests);

        //collectionBeforePaging = collectionBeforePaging
        //    .Include(c => c.AuditReviewers)
        //        .ThenInclude(c => c.Reviewer)
        //    .Include(c => c.AuditReviewers)
        //        .ThenInclude(c => c.Submitter);

        collectionBeforePaging = collectionBeforePaging.ApplySort(auditResource.OrderBy,
            _propertyMappingService.GetPropertyMapping<AuditDto, Audit>());

        if (!string.IsNullOrWhiteSpace(auditResource.AuditTemplateName))
        {
            collectionBeforePaging = collectionBeforePaging
            .Where(c => c.AuditTemplate.Name.StartsWith(auditResource.AuditTemplateName));
        }
        if (!string.IsNullOrWhiteSpace(auditResource.Title))
        {
            collectionBeforePaging = collectionBeforePaging
            .Where(c => c.Title.StartsWith(auditResource.Title));
        }
        if (auditResource.DepartmentId.HasValue)
        {
            collectionBeforePaging = collectionBeforePaging
          .Where(c => c.AuditTemplate.DepartmentId == auditResource.DepartmentId);
        }
        if (auditResource.AuditorId.HasValue)
        {
            collectionBeforePaging = collectionBeforePaging
          .Where(c => c.AuditorId == auditResource.AuditorId);
        }
        if (auditResource.auditStatus.HasValue)
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => c.Status == auditResource.auditStatus);
        }
        if (auditResource.auditType.HasValue)
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => c.AuditTemplate.AuditType == auditResource.auditType);
        }

        var allAuditList = new AuditList();
        return await allAuditList.Create(
            collectionBeforePaging,
            auditResource.Skip,
            auditResource.PageSize);

    }

    public async Task<AuditList> GetAuditForReview(AuditResource auditResource)
    {
        var collectionBeforePaging = AllIncluding(
            c => c.AuditTemplate,
            c => c.Auditor,
            c => c.NonConformances,
            c => c.CapaRequests,
            c => c.AuditReviewers)
        .Where(c => c.Status == AUDIT_STATUS.INREVIEW &&
                    c.AuditReviewers
                     .OrderByDescending(r => r.ReviewedOn)
                     .FirstOrDefault() != null &&
                    c.AuditReviewers
                     .OrderByDescending(r => r.ReviewedOn)
                     .FirstOrDefault().ReviewerId == userInfoToken.Id);

        collectionBeforePaging = collectionBeforePaging.ApplySort(auditResource.OrderBy,
            _propertyMappingService.GetPropertyMapping<AuditDto, Audit>());

        if (!string.IsNullOrWhiteSpace(auditResource.AuditTemplateName))
        {
            collectionBeforePaging = collectionBeforePaging
            .Where(c => c.AuditTemplate.Name.StartsWith(auditResource.AuditTemplateName));
        }
        if (!string.IsNullOrWhiteSpace(auditResource.Title))
        {
            collectionBeforePaging = collectionBeforePaging
            .Where(c => c.Title.StartsWith(auditResource.Title));
        }
        if (auditResource.DepartmentId.HasValue)
        {
            collectionBeforePaging = collectionBeforePaging
          .Where(c => c.AuditTemplate.DepartmentId == auditResource.DepartmentId);
        }
        if (auditResource.AuditorId.HasValue)
        {
            collectionBeforePaging = collectionBeforePaging
          .Where(c => c.AuditorId == auditResource.AuditorId);
        }
        if (auditResource.auditStatus.HasValue)
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => c.Status == auditResource.auditStatus);
        }
        if (auditResource.auditType.HasValue)
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => c.AuditTemplate.AuditType == auditResource.auditType);
        }

        var allAuditList = new AuditList();
        return await allAuditList.Create(
            collectionBeforePaging,
            auditResource.Skip,
            auditResource.PageSize);
    }
}
