﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;

namespace QualityManagement.Repository;

public class AuditTemplateList : List<AuditTemplateDto>
{
    public int Skip { get; private set; }
    public int TotalPages { get; private set; }
    public int PageSize { get; private set; }
    public int TotalCount { get; private set; }

    public AuditTemplateList(List<AuditTemplateDto> items, int count, int skip, int pageSize)
    {
        TotalCount = count;
        PageSize = pageSize;
        Skip = skip;
        TotalPages = (int)Math.Ceiling(count / (double)pageSize);
        AddRange(items);
    }
    public AuditTemplateList()
    {

    }

    public async Task<AuditTemplateList> Create(IQueryable<AuditTemplate> source, int skip, int pageSize)
    {
        var count = await GetCount(source);
        var dtoList = await GetDtos(source, skip, pageSize);
        var dtoPageList = new AuditTemplateList(dtoList, count, skip, pageSize);
        return dtoPageList;
    }

    public async Task<int> GetCount(IQueryable<AuditTemplate> source)
    {
        return await source.AsNoTracking().CountAsync();
    }

    public async Task<List<AuditTemplateDto>> GetDtos(IQueryable<AuditTemplate> source, int skip, int pageSize)
    {
        try
        {
            var auditTemplates = await source
                .Skip(skip)
                .Take(pageSize)
                .AsNoTracking()
                .Select(c => new AuditTemplateDto
                {
                    Id = c.Id,
                    Name = c.Name,
                    AuditType = c.AuditType,
                    DepartmentId = c.DepartmentId.Value,
                    Description = c.Description,
                    IconName = c.IconName,
                    TotalQuestions = c.Questions.Count,
                    AuditId = c.Audits
                       .Where(a => a.AuditTemplateId == c.Id &&
                                   (a.Status == AUDIT_STATUS.CREATED || a.Status == AUDIT_STATUS.INPROGRESS || a.Status == AUDIT_STATUS.REWORK))
                       .Select(a => a.Id)
                       .FirstOrDefault(),
                    IsAuditRunning = c.Audits.Any(a => a.Status == AUDIT_STATUS.CREATED || a.Status == AUDIT_STATUS.INPROGRESS || a.Status == AUDIT_STATUS.REWORK)
                })
                .ToListAsync();
            return auditTemplates;
        }
        catch (Exception ex)
        {
            throw new DataException("Error while getting AuditTemplate", ex);
        }
    }
}
