﻿using System;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Common.GenericRepository;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Data.Resources;
using QualityManagement.Domain;

namespace QualityManagement.Repository;
public class CapaRequestRepository(
    IUnitOfWork<QMSDbContext> uow,
    IPropertyMappingService _propertyMappingService) : GenericRepository<CapaRequest, QMSDbContext>(uow), ICapaRequestRepository
{
    public async Task<CapaRequestList> GetCapaRequest(CapaRequestResource capaRequestResource)
    {

        var collectionBeforePaging = AllIncluding(c => c.NonConformance, c => c.Audit, c => c.RaisedBY);

        collectionBeforePaging = collectionBeforePaging.ApplySort(capaRequestResource.OrderBy,
            _propertyMappingService.GetPropertyMapping<CapaRequestDto, CapaRequest>());

        if (!string.IsNullOrWhiteSpace(capaRequestResource.ReferenceNo))
        {
            collectionBeforePaging = collectionBeforePaging
            .Where(c => c.ReferenceNo.Contains(capaRequestResource.ReferenceNo));
        }
        if (!string.IsNullOrWhiteSpace(capaRequestResource.Title))
        {
            collectionBeforePaging = collectionBeforePaging
            .Where(c => c.Title.Contains(capaRequestResource.Title));
        }
        if (capaRequestResource.RaisedById.HasValue)
        {
            collectionBeforePaging = collectionBeforePaging
          .Where(c => c.RaisedById == capaRequestResource.RaisedById);
        }
        if (capaRequestResource.AuditId.HasValue)
        {
            collectionBeforePaging = collectionBeforePaging
          .Where(c => c.AuditId == capaRequestResource.AuditId);
        }
        if (capaRequestResource.NonConformanceId.HasValue)
        {
            collectionBeforePaging = collectionBeforePaging
          .Where(c => c.NonConformanceId == capaRequestResource.NonConformanceId);
        }
        if (capaRequestResource.Priority.HasValue)
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => c.Priority == capaRequestResource.Priority);
        }
        if (capaRequestResource.CapaRequestStatus.HasValue)
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => c.CapaRequestStatus == capaRequestResource.CapaRequestStatus);
        }
        if (capaRequestResource.DueStartDate.HasValue && capaRequestResource.DueEndDate.HasValue)
        {

            var startDate = capaRequestResource.DueStartDate.Value;
            var endDate = capaRequestResource.DueEndDate.Value;

            DateTime minDate = new DateTime(startDate.Year, startDate.Month, startDate.Day, 0, 0, 0);
            DateTime maxDate = new DateTime(endDate.Year, endDate.Month, endDate.Day, 23, 59, 59);

            collectionBeforePaging = collectionBeforePaging
                        .Where(c => c.DueDate >= minDate &&
                            c.DueDate <= maxDate);
        }


        var capaRequestList = new CapaRequestList();
        return await capaRequestList.Create(
            collectionBeforePaging,
            capaRequestResource.Skip,
            capaRequestResource.PageSize);


    }
    public async Task<string> GenerateCapaNumberAsync()
    {

        var lastNcNumber = await All
                .OrderByDescending(c => c.CreatedDate)
                .FirstOrDefaultAsync();

        string prefix = "CAPA";
        int year = DateTime.UtcNow.Year;
        int nextNumber = 1;

        if (lastNcNumber?.ReferenceNo is string lastNumber)
        {
            var parts = lastNumber.Split('-');
            if (parts.Length == 3 && int.TryParse(parts[2], out int lastNum))
            {
                nextNumber = lastNum + 1;
            }
        }
        return $"{prefix}-{year}-{nextNumber:D4}";
    }
}
