﻿using Microsoft.EntityFrameworkCore;
using QualityManagement.Common.GenericRepository;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Data.Resources;
using QualityManagement.Domain;
using System;
using System.Linq;
using System.Threading.Tasks;

namespace QualityManagement.Repository;

public class DocumentAuditTrailRepository(
    IPropertyMappingService _propertyMappingService,
    IUnitOfWork<QMSDbContext> uow) : GenericRepository<DocumentAuditTrail, QMSDbContext>(uow),
    IDocumentAuditTrailRepository
{

    public async Task<DocumentAuditTrailList> GetDocumentAuditTrails(DocumentResource documentResource)
    {
        var collectionBeforePaging = AllIncluding(c => c.CreatedByUser, d => d.Document, c => c.Document.Category, c => c.Category);
        collectionBeforePaging =
           collectionBeforePaging
            .IgnoreQueryFilters()
            .ApplySort(documentResource.OrderBy,
           _propertyMappingService.GetPropertyMapping<DocumentAuditTrailDto, DocumentAuditTrail>());

        if (!string.IsNullOrWhiteSpace(documentResource.Name))
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => EF.Functions.Like(c.Document.Name, $"%{documentResource.Name}%"));
        }
        if (!string.IsNullOrWhiteSpace(documentResource.DocumentNumber))
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => EF.Functions.Like(c.Document.DocumentNumber, $"{documentResource.DocumentNumber}%"));
        }
        if (!string.IsNullOrWhiteSpace(documentResource.Id))
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => c.DocumentId == Guid.Parse(documentResource.Id));
        }
        if (!string.IsNullOrWhiteSpace(documentResource.CategoryId))
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => c.CategoryId == Guid.Parse(documentResource.CategoryId));
        }
        if (!string.IsNullOrWhiteSpace(documentResource.Operation))
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => c.OperationName == Enum.Parse<DocumentOperation>(documentResource.Operation));
        }
        if (!string.IsNullOrWhiteSpace(documentResource.CreatedBy))
        {
            collectionBeforePaging = collectionBeforePaging
                .Where(c => c.CreatedBy == Guid.Parse(documentResource.CreatedBy));
        }
        var documentAuditTrailList = new DocumentAuditTrailList();
        return await documentAuditTrailList.Create(
            collectionBeforePaging,
            documentResource.Skip,
            documentResource.PageSize
            );
    }
}
