﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Data;
using QualityManagement.Data.Dto;

namespace QualityManagement.Repository;
public class AssignedEmployeeCourseList : List<EmployeeCourseDto>
{
    public AssignedEmployeeCourseList()
    {
    }

    public int Skip { get; private set; }
    public int TotalPages { get; private set; }
    public int PageSize { get; private set; }
    public int TotalCount { get; private set; }

    public AssignedEmployeeCourseList(List<EmployeeCourseDto> items, int count, int skip, int pageSize)
    {
        TotalCount = count;
        PageSize = pageSize;
        Skip = skip;
        TotalPages = (int)Math.Ceiling(count / (double)pageSize);
        AddRange(items);
    }

    public async Task<AssignedEmployeeCourseList> Create(IQueryable<EmployeeCourse> source, int skip, int pageSize, string orderBy)
    {
        var count = await GetCount(source);
        var dtoList = await GetDtos(source, skip, pageSize, orderBy);
        var dtoPageList = new AssignedEmployeeCourseList(dtoList, count, skip, pageSize);
        return dtoPageList;
    }

    public async Task<int> GetCount(IQueryable<EmployeeCourse> source)
    {
        return await source.AsNoTracking().CountAsync();
    }

    public async Task<List<EmployeeCourseDto>> GetDtos(IQueryable<EmployeeCourse> source, int skip, int pageSize, string orderBy)
    {
        if (pageSize == 0)
        {
            return await source
                .Select(c => new EmployeeCourseDto
                {
                    Id = c.Id,
                    EmployeeId = c.EmployeeId,
                    CourseId = c.CourseId,
                    CourseName = c.Course.Title,
                    Status = c.Status,
                    Link = c.Link,
                    EmployeeName = $"{c.Employee.FirstName} {c.Employee.LastName}",
                    AssignedDate = c.CreatedDate,
                    DueDate = c.DueDate,
                    CompletedDate = c.CompletedDate
                })
                .AsNoTracking()
                .ToListAsync();
        }
        var entities = await source
            .Select(c => new EmployeeCourseDto
            {
                Id = c.Id,
                EmployeeId = c.EmployeeId,
                CourseId = c.CourseId,
                CourseName = c.Course.Title,
                Status = c.Status,
                Link = c.Link,
                EmployeeName = $"{c.Employee.FirstName} {c.Employee.LastName}",
                AssignedDate = c.CreatedDate,
                DueDate = c.DueDate,
                CompletedDate = c.CompletedDate
            })
            .Skip(skip)
            .Take(pageSize)
            .AsNoTracking()
            .ToListAsync();
        return entities;
    }
}