﻿using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.GenericRepository;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Resources;
using QualityManagement.Domain;
using System;
using System.Linq;
using System.Threading.Tasks;

namespace QualityManagement.Repository
{
    public class LoginAuditRepository(
        IUnitOfWork<QMSDbContext> uow,
            ILogger<LoginAuditRepository> _logger,
            IPropertyMappingService _propertyMappingService) : GenericRepository<LoginAudit, QMSDbContext>(uow),
       ILoginAuditRepository
    {

        public async Task<LoginAuditList> GetLoginAudits(LoginAuditResource loginAuditResrouce)
        {
            var collectionBeforePaging = All;
            collectionBeforePaging =
               collectionBeforePaging.ApplySort(loginAuditResrouce.OrderBy,
               _propertyMappingService.GetPropertyMapping<LoginAuditDto, LoginAudit>());

            if (!string.IsNullOrWhiteSpace(loginAuditResrouce.UserName))
            {
                collectionBeforePaging = collectionBeforePaging
                    .Where(c => EF.Functions.Like(c.UserName, $"%{loginAuditResrouce.UserName}%"));
            }

            var loginAudits = new LoginAuditList();
            return await loginAudits.Create(
                collectionBeforePaging,
                loginAuditResrouce.Skip,
                loginAuditResrouce.PageSize
                );
        }

        public async Task LoginAudit(LoginAuditDto loginAudit)
        {
            try
            {
                Add(new LoginAudit
                {
                    Id = Guid.NewGuid(),
                    LoginTime = DateTime.UtcNow,
                    Provider = loginAudit.Provider,
                    Status = loginAudit.Status,
                    UserName = loginAudit.UserName,
                    RemoteIP = loginAudit.RemoteIP,
                    Latitude = loginAudit.Latitude,
                    Longitude = loginAudit.Longitude
                });
                await _uow.SaveAsync();
            }
            catch (Exception ex)
            {
                _logger.LogError(ex.Message);
            }
        }
        //QualityManagement method 

        public async Task SetLogoutTimeAsync(string email)
        {
            try
            {
                var lastLoginTime = await All.Where(c => c.UserName == email && !c.LogoutTime.HasValue)
              .OrderByDescending(d => d.LoginTime)
              .FirstOrDefaultAsync();

                if (lastLoginTime != null)
                {
                    lastLoginTime.LogoutTime = DateTime.UtcNow;
                    Update(lastLoginTime);
                    await _uow.SaveAsync();
                }
            }
            catch (Exception ex)
            {
                //QualityManagement
                //logger
                _logger.LogError("{Message}", ex.Message);
            }
        }
    }
}
