﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;

namespace QualityManagement.Repository;

public class NonConformanceList : List<NonConformanceDto>
{

    public int Skip { get; private set; }
    public int TotalPages { get; private set; }
    public int PageSize { get; private set; }
    public int TotalCount { get; private set; }

    public NonConformanceList()
    {
    }
    public NonConformanceList(List<NonConformanceDto> items, int count, int skip, int pageSize)
    {
        TotalCount = count;
        PageSize = pageSize;
        Skip = skip;
        TotalPages = (int)Math.Ceiling(count / (double)pageSize);
        AddRange(items);
    }
    public async Task<NonConformanceList> Create(IQueryable<NonConformance> source, int skip, int pageSize)
    {
        var count = await GetCount(source);
        var dtoList = await GetDtos(source, skip, pageSize);
        var dtoPageList = new NonConformanceList(dtoList, count, skip, pageSize);
        return dtoPageList;
    }

    public async Task<int> GetCount(IQueryable<NonConformance> source)
    {
        return await source.AsNoTracking().CountAsync();
    }

    public async Task<List<NonConformanceDto>> GetDtos(IQueryable<NonConformance> source, int skip, int pageSize)
    {
        try
        {
            var nonConformances = await source
                .Skip(skip)
                .Take(pageSize)
                .AsNoTracking()
                .Select(c => new NonConformanceDto
                {
                    Id = c.Id,
                    Title = c.Title,
                    DepartmentId = c.DepartmentId.Value,
                    Description = c.Description,
                    AuditId = c.AuditId,
                    ClosedDate = c.ClosedDate,
                    NCNumber = c.NCNumber,
                    ReportedById = c.ReportedById,
                    Severity = c.Severity,
                    Status = c.Status,
                    DepartmentName = c.Department != null ? c.Department.Name : null,
                    AuditName = c.Audit != null ? c.Audit.Title : null,
                    ReportedByName = c.ReportedBy != null ? $"{c.ReportedBy.FirstName}{c.ReportedBy.LastName}" : null,

                })
                .ToListAsync();
            return nonConformances;
        }
        catch (Exception ex)
        {
            throw new DataException("Error while getting NonConformance", ex);
        }
    }
}
