﻿using Microsoft.EntityFrameworkCore;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace QualityManagement.Repository
{
    public class NonConformanceLogList : List<NonConformanceLogDto>
    {
        public int Skip { get; private set; }
        public int TotalPages { get; private set; }
        public int PageSize { get; private set; }
        public int TotalCount { get; private set; }
        public NonConformanceLogList()
        {

        }
        public NonConformanceLogList(List<NonConformanceLogDto> items, int count, int skip, int pageSize)
        {
            TotalCount = count;
            PageSize = pageSize;
            Skip = skip;
            TotalPages = (int)Math.Ceiling(count / (double)pageSize);
            AddRange(items);
        }

        public async Task<NonConformanceLogList> Create(IQueryable<NonConformanceLog> source, int skip, int pageSize)
        {
            var count = await GetCount(source);
            var dtoList = await GetDtos(source, skip, pageSize);
            var dtoPageList = new NonConformanceLogList(dtoList, count, skip, pageSize);
            return dtoPageList;
        }
        public async Task<int> GetCount(IQueryable<NonConformanceLog> source)
        {
            return await source.AsNoTracking().CountAsync();
        }
        public async Task<List<NonConformanceLogDto>> GetDtos(IQueryable<NonConformanceLog> source, int skip, int pageSize)
        {
            try
            {
                var nonConformancesLog = await source
                    .Skip(skip)
                    .Take(pageSize)
                    .AsNoTracking()
                    .Select(c => new NonConformanceLogDto
                    {
                        Id = c.Id,
                        Title = c.Title,
                        DepartmentId = c.DepartmentId.Value,
                        LogStatus=c.LogStatus,
                        Description = c.Description,
                        AuditId = c.AuditId,
                        ClosedDate = c.ClosedDate,
                        NCNumber = c.NCNumber,
                        ReportedById = c.ReportedById,
                        Severity = c.Severity,
                        Status = c.Status,
                        DepartmentName = c.Department != null ? c.Department.Name : null,
                        AuditName = c.Audit != null ? c.Audit.Title : null,
                        ReportedByName = c.ReportedBy != null ? $"{c.ReportedBy.FirstName}{c.ReportedBy.LastName}" : null,

                    })
                    .ToListAsync();
                return nonConformancesLog;
            }
            catch (Exception ex)
            {
                throw new DataException("Error while getting NonConformance Log", ex);
            }

        }
    }
}
