﻿using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Resources;
using QualityManagement.MediatR;
using QualityManagement.MediatR.Capa;

namespace QualityManagement.API.Controllers.Capa;

/// <summary>
///  Controller For a CapaRequest
/// </summary>
[Route("api/[controller]")]
[ApiController]
[Authorize]
public class CapaRequestController(
    IMediator _mediator) : BaseController
{
    /// <summary>
    /// Controller for Create CapaRequest
    /// </summary>
    /// <param name="command"></param>
    /// <returns></returns>
    [HttpPost]
    [ClaimCheck("ADD_CAPA_REQUEST")]
    public async Task<IActionResult> CreateCapaRequest(AddCapaRequestCommand command)
    {
        var result = await _mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Controller for Update CapaRequest by id
    /// </summary>
    /// <param name="id"></param>
    /// <param name="command"></param>
    /// <returns></returns>
    [HttpPut("{id}")]
    [ClaimCheck("EDIT_CAPA_REQUEST")]
    public async Task<IActionResult> UpdateCapaRequest(Guid id, UpdateCapaRequestCommand command)

    {
        command.Id = id;
        var result = await _mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// delete CapaRequest by id
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpDelete("{id}")]
    [ClaimCheck("DELETE_CAPA_REQUEST")]
    public async Task<IActionResult> DeleteCapaRequest(Guid id)
    {
        var result = await _mediator.Send(new DeleteCapaRequestCommand { Id = id });
        return GenerateResponse(result);
    }

    /// <summary>
    /// get CapaRequest by id
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}")]
    public async Task<IActionResult> GetCapaRequest(Guid id)
    {
        var result = await _mediator.Send(new GetCapaRequestCommand { Id = id });
        return GenerateResponse(result);
    }

    /// <summary>
    /// get all CapaRequest
    /// </summary>
    /// <param name="capaRequestResource"></param>
    /// <returns></returns>
    [HttpGet]
    [ClaimCheck("VIEW_CAPA_REQUEST")]
    public async Task<IActionResult> GetCapaRequests([FromQuery] CapaRequestResource capaRequestResource)
    {
        var getCapaRequestListCommand = new GetCapaRequestListCommand()
        {
            CapaRequestResource = capaRequestResource
        };
        var result = await _mediator.Send(getCapaRequestListCommand);
        var paginationMetadata = new
        {
            totalCount = result.TotalCount,
            pageSize = result.PageSize,
            skip = result.Skip,
            totalPages = result.TotalPages
        };
        Response.Headers.Append("X-Pagination",
            Newtonsoft.Json.JsonConvert.SerializeObject(paginationMetadata));

        return Ok(result);
    }

    /// <summary>
    /// Get All Audits
    /// </summary>
    /// <returns></returns>
    [HttpGet("audits")]
    public async Task<IActionResult> GetAllAudits()
    {
        var query = new GetAllCapaRequestAuditsCommand { };
        var result = await _mediator.Send(query);
        return Ok(result);
    }

    /// <summary>
    /// Get a capaClosure Rate
    /// </summary>
    /// <returns></returns>
    [HttpGet("CapaClosureRate")]
    public async Task<IActionResult> GetClosureRate()
    {
        var result = await _mediator.Send(new GetCapaClosureRateCommand());
        return Ok(result);
    }

    /// <summary>
    /// Get a Capa OnTime Complete Rate
    /// </summary>
    /// <returns></returns>
    [HttpGet("OnTimeCapa")]
    public async Task<IActionResult> GetOnTimeCapa()
    {
        var result = await _mediator.Send(new GetCapaOnTimeCompletionCommand());
        return Ok(result);
    }

    /// <summary>
    /// Get A Capa Overview
    /// </summary>
    /// <returns></returns>
    [HttpGet("CapaOverview")]
    [Produces("application/json", "application/xml", Type = typeof(List<NameCountDto>))]
    public async Task<IActionResult> GetCapaOverview([FromQuery] GetCapaOverviewCommand command)
    {
        var result = await _mediator.Send(command);
        return Ok(result);
    }
}
