﻿using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.MediatR;

namespace QualityManagement.API.Controllers.Category;

/// <summary>
/// Category
/// </summary>
[Route("api/[controller]")]
[ApiController]
[Authorize]
public class CategoryController(IMediator _mediator) : ControllerBase
{
    /// <summary>
    /// Get Specific Category by ID.
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}")]
    [Produces("application/json", "application/xml", Type = typeof(CategoryDto))]
    public async Task<IActionResult> GetCategory(Guid id)
    {
        var getCategoryQuery = new GetCategoryQuery
        {
            Id = id
        };
        var result = await _mediator.Send(getCategoryQuery);
        return StatusCode(result.StatusCode, result);
    }
    /// <summary>
    /// Get Categories by ParentId.
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>

    [HttpGet("{id}/children")]
    [Produces("application/json", "application/xml", Type = typeof(CategoryDto))]
    public async Task<IActionResult> GetCategoriesByParentId(Guid id)
    {
        var getCategoriesByParentIdCommand = new GetCategoriesByParentIdCommand
        {
            ParentId = id
        };
        var result = await _mediator.Send(getCategoriesByParentIdCommand);
        return StatusCode(result.StatusCode, result.Data);
    }
    /// <summary>
    /// Get Categories Share by ParentId.
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}/shared/children")]
    [Produces("application/json", "application/xml", Type = typeof(CategoryDto))]
    public async Task<IActionResult> GetCategoriesSharedByParentId(Guid id)
    {
        var getCategoriesSharedByParentIdCommand = new GetCategoriesSharedByParentIdCommand
        {
            ParentId = id
        };
        var result = await _mediator.Send(getCategoriesSharedByParentIdCommand);
        return StatusCode(result.StatusCode, result.Data);
    }
    /// <summary>
    ///Get Categories Hierarchical By ChildId.
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>

    [HttpGet("{id}/hierarchical")]
    [Produces("application/json", "application/xml", Type = typeof(CategoryDto))]
    public async Task<IActionResult> GetCategoriesHierarchicalBChildId(Guid id)
    {
        var getCategoriesHierarchicalBChildIdCommand = new GetCategoriesHierarchicalBChildIdCommand
        {
            ChildId = id
        };
        var result = await _mediator.Send(getCategoriesHierarchicalBChildIdCommand);
        return StatusCode(result.StatusCode, result.Data);
    }
    /// <summary>
    ///Get Categories Hierarchical By ChildId For Window Shared.
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}/hierarchical/windowshared")]
    [Produces("application/json", "application/xml", Type = typeof(CategoryDto))]
    public async Task<IActionResult> GetCategoriesHierarchicalBChildIdForWindowShared(Guid id)
    {
        var getCategoriesHierarchicalBChildIdWindowSharedCommand = new GetCategoriesHierarchicalBChildIdWindowSharedCommand
        {
            Id = id
        };
        var result = await _mediator.Send(getCategoriesHierarchicalBChildIdWindowSharedCommand);
        return StatusCode(result.StatusCode, result.Data);
    }
    /// <summary>
    /// Get All Categories.
    /// </summary>
    /// <returns></returns>
    [HttpGet]
    [Produces("application/json", "application/xml", Type = typeof(List<CategoryDto>))]
    public async Task<IActionResult> GetCategories()
    {
        var getAllCategoryQuery = new GetAllCategoryQuery { IsParentOnly = true };
        var result = await _mediator.Send(getAllCategoryQuery);
        return Ok(result);
    }
    /// <summary>
    /// Create a Category.
    /// </summary>
    [HttpPost]
    [Produces("application/json", "application/xml", Type = typeof(CategoryDto))]
    [ClaimCheck("CREATE_CATEGORY")]
    public async Task<IActionResult> AddCategory([FromBody] AddCategoryCommand addCategoryCommand)
    {
        var result = await _mediator.Send(addCategoryCommand);
        if (result.StatusCode != 200)
        {
            return StatusCode(result.StatusCode, result);
        }
        return CreatedAtAction("GetCategory", new { id = result.Id }, result);
    }
    /// <summary>
    /// Create a Category Assign To Me.
    /// </summary>
    /// <returns></returns>
    [HttpPost("AssignToMe")]
    [Produces("application/json", "application/xml", Type = typeof(CategoryDto))]
    public async Task<IActionResult> AddCategoryAssignToMe([FromBody] AddCategoryAssignToMeCommand addCategoryAssignToMeCommand)
    {
        var result = await _mediator.Send(addCategoryAssignToMeCommand);
        if (result.StatusCode != 200)
        {
            return StatusCode(result.StatusCode, result);
        }
        return CreatedAtAction("GetCategory", new { id = result.Id }, result);
    }
    /// <summary>
    ///Update Category Assign To Me.
    /// </summary>
    /// <returns></returns>
    [HttpPut("AssignToMe/{Id}")]
    [Produces("application/json", "application/xml", Type = typeof(CategoryDto))]
    public async Task<IActionResult> UpdateCategoryAssignToMe(Guid Id, [FromBody] UpdateCategoryAssignToMeCommand category)
    {
        category.Id = Id;
        var result = await _mediator.Send(category);
        return StatusCode(result.StatusCode, result);

    }
    /// <summary>
    /// Update Category.
    /// </summary>
    /// <param name="Id"></param>
    /// <param name="category"></param>
    /// <returns></returns>
    [HttpPut("{Id}")]
    [Produces("application/json", "application/xml", Type = typeof(CategoryDto))]
    [ClaimCheck("EDIT_CATEGORY")]
    public async Task<IActionResult> UpdateCategory(Guid Id, [FromBody] UpdateCategoryCommand category)
    {
        category.Id = Id;
        var result = await _mediator.Send(category);
        return StatusCode(result.StatusCode, result);

    }
    /// <summary>
    /// Delete Category.
    /// </summary>
    /// <param name="Id"></param>
    /// <returns></returns>
    [HttpDelete("{Id}")]
    [ClaimCheck("DELETE_CATEGORY")]
    public async Task<IActionResult> DeleteCategory(Guid Id)
    {
        var deleteCategoryCommand = new DeleteCategoryCommand
        {
            Id = Id
        };
        var result = await _mediator.Send(deleteCategoryCommand);
        return StatusCode(result.StatusCode, result);
    }
    /// <summary>
    /// Delete Category.
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}/subcategories")]
    [Produces("application/json", "application/xml", Type = typeof(List<CategoryDto>))]
    public async Task<IActionResult> GetSubCategories(Guid id)
    {
        var getSubCategoriesQuery = new GetSubCategoriesQuery { Id = id };
        var result = await _mediator.Send(getSubCategoriesQuery);
        return Ok(result);
    }
    /// <summary>
    /// Get All Descendants Using CTE By ParentId.
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}/AllDescendantsChilds")]
    [Produces("application/json", "application/xml", Type = typeof(List<CategoryDto>))]
    public async Task<IActionResult> GetAllDescendantsUsingCTEByParentId(Guid id)
    {
        var getAllDescendantsUsingCTEByParentIdCommand = new GetAllDescendantsUsingCTEByParentIdCommand { ParentId = id };
        var result = await _mediator.Send(getAllDescendantsUsingCTEByParentIdCommand);
        return Ok(result.Data);
    }
    /// <summary>
    /// Get All Categories For a DropDown.
    /// </summary>
    /// <returns></returns>
    [HttpGet("dropdown")]
    [Produces("application/json", "application/xml", Type = typeof(List<CategoryDto>))]
    public async Task<IActionResult> GetAllCategoriesForDropDown()
    {
        var getAllCategoryQuery = new GetAllCategoryQuery { IsParentOnly = false };
        var result = await _mediator.Send(getAllCategoryQuery);
        return Ok(result);
    }
    /// <summary>
    /// Get All Assign To Me Categories For a DropDown.
    /// </summary>
    /// <returns></returns>
    [HttpGet("AssignToMe/dropdown")]
    [Produces("application/json", "application/xml", Type = typeof(List<CategoryDto>))]
    public async Task<IActionResult> GetAllAssignToMeCategoriesForDropDown()
    {
        var getAllAssignToMeCategoryQuery = new GetAllAssignToMeCategoryQuery { IsParentOnly = false };
        var result = await _mediator.Send(getAllAssignToMeCategoryQuery);
        return Ok(result);
    }
    /// <summary>
    /// Get All Assign To Me Categories Search For a DropDown.
    /// </summary>
    /// <returns></returns>
    [HttpGet("AssignToMe/searchdropdown")]
    [Produces("application/json", "application/xml", Type = typeof(List<CategoryDto>))]
    public async Task<IActionResult> GetAllAssignToMeCategoriesSearchForDropDown()
    {
        var getCategoriesForListDocumentCommand = new GetCategoriesForListDocumentCommand { };
        var result = await _mediator.Send(getCategoriesForListDocumentCommand);
        return Ok(result);
    }
    /// <summary>
    /// Create Archive Categories.
    /// </summary>
    /// <param name="Id"></param>
    /// <returns></returns>
    [HttpPost("{Id}/archive")]
    [ClaimCheck("ARCHIVE_FOLDER", "ALL_ARCHIVE_FOLDER")]
    public async Task<IActionResult> ArchiveCategory(Guid Id)
    {
        var archiveCategoryCommand = new ArchiveCategoryCommand
        {
            CategoryId = Id
        };
        var result = await _mediator.Send(archiveCategoryCommand);
        return StatusCode(result.StatusCode, result);
    }
    /// <summary>
    /// Create Archive Assign Folder.
    /// </summary>
    /// <param name="Id"></param>
    /// <returns></returns>
    [HttpPost("{Id}/AssignFolders/archive")]
    [ClaimCheck("ASSIGNED_ARCHIVE_FOLDER")]
    public async Task<IActionResult> ArchiveAssignFolder(Guid Id)
    {
        var archieveCategoryFromAssignUserCommand = new ArchieveCategoryFromAssignUserCommand
        {
            CategoryId = Id
        };
        var result = await _mediator.Send(archieveCategoryFromAssignUserCommand);
        return StatusCode(result.StatusCode, result);
    }
    /// <summary>
    /// Get Archive Folders.
    /// </summary>
    /// <param name="Id"></param>
    /// <returns></returns>
    [HttpGet("{Id}/archive")]
    public async Task<IActionResult> GetArchieveFolders(Guid Id)
    {
        var getArchiveCategoriesByParentIdCommand = new GetArchiveCategoriesByParentIdCommand
        {
            ParentId = Id
        };
        var result = await _mediator.Send(getArchiveCategoriesByParentIdCommand);
        return Ok(result.Data);
    }
    /// <summary>
    /// Create Restore Categories.
    /// </summary>
    /// <param name="Id"></param>
    /// <returns></returns>
    [HttpPost("{Id}/restore")]
    [ClaimCheck("RESTORE_FOLDER")]
    public async Task<IActionResult> RestoreCategory(Guid Id)
    {
        var restoreCategoryCommand = new RestoreCategoryCommand
        {
            CategoryId = Id
        };
        var result = await _mediator.Send(restoreCategoryCommand);
        return StatusCode(result.StatusCode, result);
    }
}
