﻿using System;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Dto.Document;
using QualityManagement.MediatR;
using QualityManagement.MediatR.Capa;

namespace QualityManagement.API.Controllers.Complaint
{
    /// <summary>
    /// Controller for Complaint Attachment
    /// </summary>
    /// <param name="_mediator"></param>
    [Route("api/[controller]")]
    [ApiController]
    [Authorize]
    public class ComplaintAttachmentController(IMediator _mediator) : BaseController
    {
        /// <summary>
        /// create a ComplaintAttachment
        /// </summary>
        /// <param name="command"></param>
        /// <returns></returns>
        [HttpPost]
        [ClaimCheck("ADD_COMPLAINT_ATTACHMENT")]
        public async Task<IActionResult> AddComplaintAttachment(AddComplaintAttachmentCommand command)
        {
            var result = await _mediator.Send(command);
            return GenerateResponse(result);
        }


        /// <summary>
        /// Delete  RiskAttchament by RiskAttachment id
        /// </summary>
        /// <param name="Id"></param>
        /// <returns></returns>
        [HttpDelete("{Id}")]
        [ClaimCheck("DELETE_COMPLAINT_ATTACHMENT")]
        public async Task<IActionResult> DeleteComplaintAttachment(Guid Id)
        {
            var deleteRiskAttachment = new DeleteComplaintAttachmentCommand { Id = Id };
            var result = await _mediator.Send(deleteRiskAttachment);
            return GenerateResponse(result);
        }


        /// <summary>
        /// Get all Complaint Attachment by Risk id
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        [HttpGet("{id}")]
        [ClaimCheck("VIEW_COMPLAINT_ATTACHMENT")]
        public async Task<IActionResult> GetAllComplaintAttachment(Guid id)
        {
            var result = await _mediator.Send(new GetAllComplaintAttachmentCommand() { ComplaintId = id });
            return GenerateResponse(result);
        }



        /// <summary>
        /// upload A ComplaintAttachment chunk
        /// </summary>
        /// <param name="command"></param>
        /// <returns></returns>
        [HttpPost("Chunk")]
        public async Task<IActionResult> UploadComplaintAttachmentChunk([FromForm] UploadComplaintAttachmentChunkCommand command)
        {
            var result = await _mediator.Send(command);
            return GenerateResponse(result);
        }


        /// <summary>
        /// Mark all chunks of a Complaint Attachment  as uploaded.
        /// </summary>
        /// <param name="markComplaintAttachmentChunksUploadedCommand"></param>
        /// <returns></returns>
        [HttpPost("Chunk/uploadStatus")]
        [Produces("application/json", "application/xml", Type = typeof(ComplaintAttachmentChunkStatus))]
        public async Task<IActionResult> MarkComplaintAttachmentAllChunkUpload(MarkComplaintAttachmentChunksUploadedCommand markComplaintAttachmentChunksUploadedCommand)
        {
            var result = await _mediator.Send(markComplaintAttachmentChunksUploadedCommand);
            return GenerateResponse(result);
        }

        /// <summary>
        /// Get Download Complaint Attachment chunk.
        /// </summary>
        /// <param name="complaintAttachmentId"></param>
        /// <param name="chunkIndex"></param>
        /// <returns></returns>
        [HttpGet("{complaintAttachmentId}/chunk/{chunkIndex}/download")]
        public async Task<IActionResult> DownloadRiskAttachment(Guid complaintAttachmentId, int chunkIndex)
        {
            var result = new DownloadComplaintAttachmentCommand
            {
                ComplaintAttachmentId = complaintAttachmentId,
                ChunkIndex = chunkIndex
            };

            var downloadAttachmentResponse = await _mediator.Send(result);
            if (!downloadAttachmentResponse.Success)
            {
                return GenerateResponse(downloadAttachmentResponse);
            }

            return Ok(downloadAttachmentResponse.Data);
        }

        /// <summary>
        /// Gets the document token.
        /// </summary>
        /// <param name="id">The identifier.</param>
        /// <returns></returns>
        [HttpGet("{id}/token")]
        [AllowAnonymous]
        public async Task<IActionResult> GetDocumentToken(Guid id)
        {
            var result = await _mediator.Send(new GetDocumentTokenByDocumentIdCommand { Id = id });
            return Ok(new { result = result.Data });
        }

        /// <summary>
        /// Get Document File By Token
        /// </summary>
        /// <param name="id"></param>
        /// <param name="officeViewerRequest"></param>
        /// <returns></returns>
        [HttpGet("{id}/officeviewer")]
        [AllowAnonymous]
        public async Task<IActionResult> GetDocumentFileByToken(Guid id, [FromQuery] OfficeViewerRequest officeViewerRequest)
        {
            var command = new DownloadComplaintAttachmentChunkCommand
            {
                Id = id,
                Token = officeViewerRequest.Token
            };
            var downloadDocumentResponse = await _mediator.Send(command);
            if (!downloadDocumentResponse.Success)
            {
                return GenerateResponse(downloadDocumentResponse);
            }
            var downloadDocument = downloadDocumentResponse.Data;
            return File(downloadDocument.Data, downloadDocument.ContentType, downloadDocument.FileName);
        }
    }

}

