﻿using System;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Resources;
using QualityManagement.MediatR;

namespace QualityManagement.API.Controllers
{
    /// <summary>
    /// Controller for Complaint
    /// </summary>
    [Route("api/[controller]")]
    [ApiController]
    [Authorize]
    public class ComplaintController(IMediator _mediator) : BaseController
    {
        /// <summary>
        /// Create A Complaint
        /// </summary>
        /// <param name="command"></param>
        /// <returns></returns>
        [HttpPost]
        [ClaimCheck("ADD_COMPLAINT")]
        public async Task<IActionResult> CreateComplaint(AddComplaintCommand command)
        {
            var result = await _mediator.Send(command);
            return GenerateResponse(result);
        }

        /// <summary>
        /// Update Complaint by complaint Id
        /// </summary>
        /// <param name="id"></param>
        /// <param name="command"></param>
        /// <returns></returns>
        [HttpPut("{id}")]
        [ClaimCheck("EDIT_COMPLAINT")]
        public async Task<IActionResult> UpdateComplaint(Guid id, UpdateComplaintCommand command)
        {
            command.Id = id;
            var result = await _mediator.Send(command);
            return GenerateResponse(result);

        }

        /// <summary>
        /// Delete Complaint by Complaint Id
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        [HttpDelete("{id}")]
        [ClaimCheck("DELETE_COMPLAINT")]
        public async Task<IActionResult> DeleteComplaint(Guid id)
        {
            var result = await _mediator.Send(new DeleteComplaintCommand() { Id = id });
            return GenerateResponse(result);
        }


        /// <summary>
        /// Get Comaplaint by Comaplaint Id
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        [HttpGet("{id}")]
        public async Task<IActionResult> GetComplaint(Guid id)
        {
            var result = await _mediator.Send(new GetComplaintCommand() { Id = id });
            return GenerateResponse(result);
        }


        /// <summary>
        /// get all Complaints
        /// </summary>
        /// <param name="complaintResource"></param>
        /// <returns></returns>
        [HttpGet]
        [ClaimCheck("VIEW_COMPLAINT")]
        public async Task<IActionResult> GetCapaRequests([FromQuery] ComplaintResource complaintResource)
        {
            var getComplaintResource = new GetComplaintListCommand()
            {
                ComplaintResource = complaintResource
            };
            var result = await _mediator.Send(getComplaintResource);
            var paginationMetadata = new
            {
                totalCount = result.TotalCount,
                pageSize = result.PageSize,
                skip = result.Skip,
                totalPages = result.TotalPages
            };
            Response.Headers.Append("X-Pagination",
                Newtonsoft.Json.JsonConvert.SerializeObject(paginationMetadata));

            return Ok(result);
        }

        /// <summary>
        /// get all Asign complaints
        /// </summary>
        /// <returns></returns>
        [HttpGet("assigned")]
        [AllowAnonymous]
        public async Task<IActionResult> GetAllAsignCapaComplaints()
        {
            var result = await _mediator.Send(new GetAllAsignComplaintCommand());
            return GenerateResponse(result);
        }

    }
}
