﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Text.Json;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Dto.Document;
using QualityManagement.Data.Resources;
using QualityManagement.MediatR;
using QualityManagement.Repository;

namespace QualityManagement.API.Controllers.Document;

/// <summary>
/// Document
/// </summary>
[Route("api")]
[ApiController]
[Authorize]
public class DocumentController(IMediator _mediator) : BaseController
{
    /// <summary>
    /// Get Archive Document 
    /// </summary>
    /// <param name="documentResource"></param>
    /// <returns></returns>
    [HttpGet("Documents/archive")]
    [Produces("application/json", "application/xml", Type = typeof(DocumentList))]
    public async Task<IActionResult> GetArchieveDocuments([FromQuery] DocumentResource documentResource)
    {
        documentResource.IsArchive = true;
        var getAllArchiveDocumentQuery = new GetAllArchiveDocumentQuery
        {
            DocumentResource = documentResource
        };
        var result = await _mediator.Send(getAllArchiveDocumentQuery);

        var paginationMetadata = new
        {
            totalCount = result.TotalCount,
            pageSize = result.PageSize,
            skip = result.Skip,
            totalPages = result.TotalPages
        };

        Response.Headers.Append("X-Pagination",
            JsonSerializer.Serialize(paginationMetadata));

        return Ok(result);
    }
    /// <summary>
    /// Get Content By OCR
    /// </summary>
    /// <param name="oCRContentExtractorCommand"></param>
    /// <returns></returns>
    [HttpPost("Documents/ocr_content_extractor")]
    [Produces("application/json", "application/xml", Type = typeof(DocumentList))]
    [ClaimCheck("OCR_CONTENT_EXTRACTOR")]
    public async Task<IActionResult> GetContentByOCR([FromForm] OCRContentExtractorCommand oCRContentExtractorCommand)
    {
        var result = await _mediator.Send(oCRContentExtractorCommand);
        if (!result.Success)
        {
            return StatusCode(result.StatusCode, result.Errors);
        }
        return StatusCode(result.StatusCode, result.Data);
    }
    /// <summary>
    /// Check Document Store As Chunk
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("Document/{id}/isChunked")]
    [Produces("application/json", "application/xml", Type = typeof(bool))]
    [AllowAnonymous]
    public async Task<IActionResult> checkDocumentStoreAsChunk(Guid id)
    {

        var checkDocumentStoreAsChunkCommand = new CheckDocumentStoreAsChunkCommand
        {
            DocumentId = id
        };
        var result = await _mediator.Send(checkDocumentStoreAsChunkCommand);

        return Ok(result.Data);
    }
    /// <summary>
    /// Get Document Summary
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("Document/{id}/summary")]
    [Produces("application/json", "application/xml", Type = typeof(bool))]
    [ClaimCheck("ALL_GET_DOCUMENT_SUMMARY", "ASSIGNED_GET_DOCUMENT_SUMMARY")]
    public async Task<IActionResult> getDocumentSummary(Guid id)
    {
        var getDocumentSummaryCommand = new GetDocumentSummaryCommand
        {
            DocumentId = id
        };
        var result = await _mediator.Send(getDocumentSummaryCommand);
        if (!result.Success)
        {
            return StatusCode(result.StatusCode, result.Errors);
        }
        return Ok(result.Data);
    }
    /// <summary>
    /// Get Document Shared UsersRoles
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("Document/{id}/sharedUsersRoles", Name = "GetDocumentSharedUsersRoles")]
    [Produces("application/json", "application/xml", Type = typeof(DocumentDto))]
    public async Task<IActionResult> GetDocumentSharedUsersRoles(Guid id)
    {
        var getDocumentQuery = new GetDocumentQuery
        {
            Id = id
        };
        var response = await _mediator.Send(getDocumentQuery);
        if (!response.Success)
        {
            return StatusCode(response.StatusCode, response.Errors);
        }
        var result = response.Data;
        return StatusCode(result.StatusCode, result);
    }
    /// <summary>
    /// Get Document Details By ID
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("Document/{id}/Detail", Name = "GetDocumentDetailById")]
    [Produces("application/json", "application/xml", Type = typeof(DocumentDto))]
    public async Task<IActionResult> GetDocumentDetailById(Guid id)
    {
        var getDocumentDetailById = new GetDocumentDetailById
        {
            Id = id
        };
        var response = await _mediator.Send(getDocumentDetailById);

        return Ok(response);
    }
    [HttpGet("Document/ArchiveRetentionPeriod")]
    public async Task<IActionResult> GetArchiveRetentionPeriod()
    {
        var getArchiveRetentionCommand = new GetArchiveRetentionCommand();
        var archiveRetention = await _mediator.Send(getArchiveRetentionCommand);
        return Ok(archiveRetention);
    }

    /// <summary>
    /// Get Document By Id
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("Document/{id}", Name = "GetDocument")]
    [Produces("application/json", "application/xml", Type = typeof(DocumentDto))]
    public async Task<IActionResult> GetDocument(Guid id)
    {
        var getDocumentQuery = new GetDocumentQuery
        {
            Id = id
        };
        var response = await _mediator.Send(getDocumentQuery);
        if (!response.Success)
        {
            return StatusCode(response.StatusCode, response.Errors);
        }
        var result = response.Data;
        return StatusCode(result.StatusCode, result);
    }
    /// <summary>
    ///  Deep Search
    /// </summary>
    /// <param name="searchQuery"></param>
    /// <returns></returns>
    [HttpGet("Document/deepSearch", Name = "DeepSearch")]
    [Produces("application/json", "application/xml", Type = typeof(List<DocumentDto>))]
    [ClaimCheck("DEEP_SEARCH")]
    public async Task<IActionResult> DeepSearch(string searchQuery)
    {
        var deepSearchCommand = new DeepSearchCommand
        {
            SearchQuery = searchQuery
        };
        var response = await _mediator.Send(deepSearchCommand);
        if (!response.Success)
        {
            return StatusCode(response.StatusCode, response.Errors);
        }
        return Ok(response.Data);
    }
    /// <summary>
    /// Remove Page Indexing
    /// </summary>
    /// <param name="Id"></param>
    /// <returns></returns>
    [HttpPost("Document/{id}/remove/pageindexing", Name = "RemovePageIndexing")]
    [Produces("application/json", "application/xml", Type = typeof(List<DocumentDto>))]
    [ClaimCheck("REMOVE_DOCUMENT_INDEX")]
    public async Task<IActionResult> RemovePageIndexing(Guid Id)
    {
        var removePageIndexingCommand = new RemovePageIndexingCommand
        {
            DocumentId = Id
        };
        var response = await _mediator.Send(removePageIndexingCommand);
        if (!response.Success)
        {
            return StatusCode(response.StatusCode, response.Errors);
        }
        return Ok(response);
    }
    /// <summary>
    /// Add Page Indexing
    /// </summary>
    /// <param name="Id"></param>
    /// <returns></returns>
    [HttpPost("Document/{id}/add/pageindexing", Name = "AddPageIndexing")]
    [Produces("application/json", "application/xml", Type = typeof(List<DocumentDto>))]
    [ClaimCheck("ADD_DOCUMENT_INDEX")]
    public async Task<IActionResult> AddPageIndexing(Guid Id)
    {
        var addPageIndexingCommand = new AddPageIndexingCommand
        {
            DocumentId = Id
        };
        var response = await _mediator.Send(addPageIndexingCommand);
        if (!response.Success)
        {
            return StatusCode(response.StatusCode, response.Errors);
        }
        return Ok(response);
    }

    /// <summary>
    /// Get All Documents
    /// </summary>
    /// <param name="documentResource"></param>
    /// <returns></returns>
    [HttpGet("Documents")]
    [Produces("application/json", "application/xml", Type = typeof(DocumentList))]
    [ClaimCheck("ALL_VIEW_DOCUMENTS")]
    public async Task<IActionResult> GetDocuments([FromQuery] DocumentResource documentResource)
    {
        documentResource.IsArchive = false;
        var getAllDocumentQuery = new GetAllDocumentQuery
        {
            DocumentResource = documentResource
        };
        var result = await _mediator.Send(getAllDocumentQuery);

        var paginationMetadata = new
        {
            totalCount = result.TotalCount,
            pageSize = result.PageSize,
            skip = result.Skip,
            totalPages = result.TotalPages
        };

        Response.Headers.Append("X-Pagination",
            JsonSerializer.Serialize(paginationMetadata));

        return Ok(result);
    }

    /// <summary>
    /// Add a document.
    /// </summary>
    /// <param name="addDocumentCommand"></param>
    /// <returns></returns>
    [HttpPost("Document")]
    [DisableRequestSizeLimit]
    [Produces("application/json", "application/xml", Type = typeof(DocumentDto))]
    [ClaimCheck("ALL_CREATE_DOCUMENT")]
    public async Task<IActionResult> AddDocument([FromForm] AddDocumentCommand addDocumentCommand)
    {
        var result = await _mediator.Send(addDocumentCommand);
        if (!result.Success)
        {
            return StatusCode(result.StatusCode, result.Errors);
        }
        return CreatedAtAction("GetDocument", new { id = result.Data.Id }, result.Data);
    }
    /// <summary>
    /// Add Document Window Shared
    /// </summary>
    /// <param name="addDocumentCommand"></param>
    /// <returns></returns>
    [HttpPost("Document/WindowShared")]
    [DisableRequestSizeLimit]
    [Produces("application/json", "application/xml", Type = typeof(DocumentDto))]
    [ClaimCheck("ALL_CREATE_DOCUMENT")]
    public async Task<IActionResult> AddDocumentWindowShared([FromForm] AddDocumentWindowSharedCommand addDocumentCommand)
    {
        var result = await _mediator.Send(addDocumentCommand);
        if (!result.Success)
        {
            return StatusCode(result.StatusCode, result.Errors);
        }
        return CreatedAtAction("GetDocument", new { id = result.Data.Id }, result.Data);
    }
    /// <summary>
    /// Upload Document Editor
    /// </summary>
    /// <param name="addDocumenFromEditorCommand"></param>
    /// <returns></returns>
    [HttpPost("Document/upload/image")]
    [DisableRequestSizeLimit]
    [Produces("application/json", "application/xml", Type = typeof(DocumentDto))]
    public async Task<IActionResult> UploadDocumentEditor([FromForm] AddDocumenFromEditorCommand addDocumenFromEditorCommand)
    {
        var result = await _mediator.Send(addDocumenFromEditorCommand);
        if (!result.Success)
        {
            return StatusCode(result.StatusCode, result.Errors);
        }
        return StatusCode(result.StatusCode, result.Data);
    }
    /// <summary>
    /// Upload Document Chunk
    /// </summary>
    /// <param name="uploadDocumentChunkCommand"></param>
    /// <returns></returns>

    [HttpPost("Document/chunk/upload")]
    [DisableRequestSizeLimit]
    [Produces("application/json", "application/xml", Type = typeof(DocumentDto))]
    [ClaimCheck("ALL_CREATE_DOCUMENT")]
    public async Task<IActionResult> UploadDocumentChunk([FromForm] UploadDocumentChunkCommand uploadDocumentChunkCommand)
    {
        var result = await _mediator.Send(uploadDocumentChunkCommand);
        if (!result.Success)
        {
            return StatusCode(result.StatusCode, result.Errors);
        }
        return StatusCode(result.StatusCode, result);
    }
    /// <summary>
    /// Add Document Chunk
    /// </summary>
    /// <param name="addDocumentChunkCommand"></param>
    /// <returns></returns>
    [HttpPost("Document/chunk")]
    [Produces("application/json", "application/xml", Type = typeof(DocumentDto))]
    [ClaimCheck("ALL_CREATE_DOCUMENT")]
    public async Task<IActionResult> AddDocumentChunk(AddDocumentChunkCommand addDocumentChunkCommand)
    {
        var result = await _mediator.Send(addDocumentChunkCommand);
        if (!result.Success)
        {
            return StatusCode(result.StatusCode, result.Errors);
        }
        return CreatedAtAction("GetDocument", new { id = result.Data.Id }, result.Data);
    }

    /// <summary>
    /// Add Document Chunk Windows Shared
    /// </summary>
    /// <param name="addDocumentChunkWindowSharedCommand"></param>
    /// <returns></returns>
    [HttpPost("Document/chunk/windowshared")]
    [Produces("application/json", "application/xml", Type = typeof(DocumentDto))]
    [ClaimCheck("ALL_CREATE_DOCUMENT")]
    public async Task<IActionResult> AddDocumentChunkWindowShared(AddDocumentChunkWindowSharedCommand addDocumentChunkWindowSharedCommand)
    {
        var result = await _mediator.Send(addDocumentChunkWindowSharedCommand);
        if (!result.Success)
        {
            return StatusCode(result.StatusCode, result.Errors);
        }
        return CreatedAtAction("GetDocument", new { id = result.Data.Id }, result.Data);
    }

    /// <summary>
    /// Mark Document All Chunk Upload
    /// </summary>
    /// <param name="markDocumentChunksUploadedStatusCommand"></param>
    /// <returns></returns>
    [HttpPost("Document/chunk/uploadStatus")]
    [Produces("application/json", "application/xml", Type = typeof(DocumentDto))]
    [ClaimCheck("ALL_CREATE_DOCUMENT")]
    public async Task<IActionResult> MarkDocumentAllChunkUpload(MarkDocumentChunksUploadedStatusCommand markDocumentChunksUploadedStatusCommand)
    {
        var result = await _mediator.Send(markDocumentChunksUploadedStatusCommand);
        if (!result.Success)
        {
            return StatusCode(result.StatusCode, result.Errors);
        }
        return Ok(result.Data);
    }

    /// <summary>
    /// add Document To me 
    /// </summary>
    /// <param name="addDocumentCommand"></param>
    /// <returns></returns>
    [HttpPost("Document/assign")]
    [DisableRequestSizeLimit]
    [Produces("application/json", "application/xml", Type = typeof(DocumentDto))]
    [ClaimCheck("ALL_CREATE_DOCUMENT", "ASSIGNED_CREATE_DOCUMENT")]
    public async Task<IActionResult> AddDocumentToMe([FromForm] AddDocumentToMeCommand addDocumentCommand)
    {
        var result = await _mediator.Send(addDocumentCommand);
        if (!result.Success)
        {
            return StatusCode(result.StatusCode, result.Errors);
        }
        return CreatedAtAction("GetDocument", new { id = result.Data.Id }, result.Data);
    }
    /// <summary>
    /// Upload document.
    /// </summary>
    /// <param name="Id"></param>
    /// <param name="updateDocumentCommand"></param>
    /// <returns></returns>
    [HttpPut("Document/{Id}")]
    [Produces("application/json", "application/xml", Type = typeof(DocumentDto))]
    [ClaimCheck("ALL_EDIT_DOCUMENT", "ASSIGNED_EDIT_DOCUMENT")]
    public async Task<IActionResult> UpdateDocument(Guid Id, UpdateDocumentCommand updateDocumentCommand)
    {
        updateDocumentCommand.Id = Id;
        var result = await _mediator.Send(updateDocumentCommand);
        return StatusCode(result.StatusCode, result);
    }
    /// <summary>
    /// Delete Document.
    /// </summary>
    /// <param name="Id"></param>
    /// <returns></returns>
    [HttpDelete("Document/{Id}")]
    [ClaimCheck("DELETE_DOCUMENT")]
    public async Task<IActionResult> DeleteDocument(Guid Id)
    {
        var deleteDocumentCommand = new DeleteDocumentCommand
        {
            Id = Id
        };
        var result = await _mediator.Send(deleteDocumentCommand);
        return StatusCode(result.StatusCode, result);
    }

    /// <summary>
    /// Archive Document.
    /// </summary>
    /// <param name="Id"></param>
    /// <returns></returns>
    [HttpPost("Document/{Id}/archive")]
    [ClaimCheck("ALL_ARCHIVE_DOCUMENT", "ASSIGNED_ARCHIVE_DOCUMENT")]
    public async Task<IActionResult> ArchiveDocument(Guid Id)
    {
        var archiveDocumentCommand = new ArchiveDocumentCommand
        {
            DocumentId = Id
        };
        var result = await _mediator.Send(archiveDocumentCommand);
        return StatusCode(result.StatusCode, result);
    }

    /// <summary>
    /// Restore Document.
    /// </summary>
    /// <param name="Id"></param>
    /// <returns></returns>
    [HttpPost("Document/{Id}/restore")]
    [ClaimCheck("RESTORE_DOCUMENT")]
    public async Task<IActionResult> RestoreDocument(Guid Id)
    {
        var archiveDocumentCommand = new RestoreDocumentCommand
        {
            DocumentId = Id
        };
        var result = await _mediator.Send(archiveDocumentCommand);
        return StatusCode(result.StatusCode, result);
    }

    /// <summary>
    /// Get Download Document.
    /// </summary>
    /// <param name="id"></param>
    /// <param name="isVersion"></param>
    /// <returns></returns>
    [HttpGet("Document/{id}/download")]
    public async Task<IActionResult> DownloadDocument(Guid id, bool isVersion)
    {
        var commnad = new DownloadDocumentCommand
        {
            Id = id,
            IsVersion = isVersion
        };

        var downloadDocumentResponse = await _mediator.Send(commnad);
        if (!downloadDocumentResponse.Success)
        {
            return GenerateResponse(downloadDocumentResponse);
        }

        var downloadDocument = downloadDocumentResponse.Data;

        return File(downloadDocument.Data, downloadDocument.ContentType, downloadDocument.FileName);
    }

    /// <summary>
    /// Get Document Chunk.
    /// </summary>
    /// <param name="id"></param>
    /// <param name="isVersion"></param>
    /// <returns></returns>
    [HttpGet("Document/{id}/chunks")]
    public async Task<IActionResult> DocumentChunks(Guid id, bool isVersion)
    {
        var commnad = new DocumentChunksCommand
        {
            DocumentId = id,
            IsVersion = isVersion
        };

        var documentChunks = await _mediator.Send(commnad);
        if (!documentChunks.Success)
        {
            return StatusCode(documentChunks.StatusCode, documentChunks.Errors);
        }
        return Ok(documentChunks.Data);
    }

    /// <summary>
    /// Get File Chunk.
    /// </summary>
    /// <param name="documentVersionId"></param>
    /// <param name="chunkIndex"></param>
    /// <returns></returns>
    [HttpGet("Document/{documentVersionId}/chunk/{chunkIndex}/download")]
    public async Task<IActionResult> GetFileChunk(Guid documentVersionId, int chunkIndex)
    {
        var command = new GetDocumentChunkCommand
        {
            DocumentVersionId = documentVersionId,
            ChunkIndex = chunkIndex
        };

        var documentChunkFile = await _mediator.Send(command);
        if (!documentChunkFile.Success)
        {
            return GenerateResponse(documentChunkFile);
        }

        return Ok(documentChunkFile.Data);
    }

    /// <summary>
    /// Get Document File By Token
    /// </summary>
    /// <param name="id"></param>
    /// <param name="officeViewerRequest"></param>
    /// <returns></returns>
    [HttpGet("Document/{id}/officeviewer")]
    [AllowAnonymous]
    public async Task<IActionResult> GetDocumentFileByToken(Guid id, [FromQuery] OfficeViewerRequest officeViewerRequest)
    {
        var getDocumentTokenCommand = new GetDocumentPathByTokenCommand
        {
            Id = id,
            Token = officeViewerRequest.Token,
            IsPublic = officeViewerRequest.IsPublic,
            IsFileRequest = officeViewerRequest.IsFileRequest
        };

        var result = await _mediator.Send(getDocumentTokenCommand);
        if (!result)
        {
            return NotFound();
        }

        if (officeViewerRequest.IsPublic)
        {
            var downloadSharedDocumentComamnd = new DownloadSharedDocumentCommand
            {
                Code = id.ToString(),
                Password = officeViewerRequest.Password
            };

            var downloadResult = await _mediator.Send(downloadSharedDocumentComamnd);
            if (!downloadResult.Success)
            {
                return GenerateResponse(downloadResult);
            }

            var sharedDocument = downloadResult.Data;
            return File(sharedDocument.Data, sharedDocument.ContentType, sharedDocument.FileName);
        }
        if (officeViewerRequest.IsFileRequest)
        {
            var downloadFileRequestDocumentCommand = new DownloadFileRequestDocumentCommand
            {
                Id = id
            };

            var downloadResult = await _mediator.Send(downloadFileRequestDocumentCommand);
            if (!downloadResult.Success)
            {
                return GenerateResponse(downloadResult);
            }
            var sharedDocument = downloadResult.Data;
            return File(sharedDocument.Data, sharedDocument.ContentType, sharedDocument.FileName);
        }

        var commnad = new DownloadDocumentChunkCommand
        {
            Id = id,
            DocumentVersionId = officeViewerRequest.DocumentVersionId
        };

        var downloadDocumentResponse = await _mediator.Send(commnad);

        if (!downloadDocumentResponse.Success)
        {
            return GenerateResponse(downloadDocumentResponse);
        }

        var downloadDocument = downloadDocumentResponse.Data;
        return File(downloadDocument.Data, downloadDocument.ContentType, downloadDocument.FileName);
    }

    /// <summary>
    /// get IS Download Flag
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("Document/{id}/isDownloadFlag")]
    [AllowAnonymous]
    public async Task<ActionResult> GetIsDownloadFlag(Guid id)
    {
        var deleteDocumentTokenCommand = new GetIsDownloadFlagQuery
        {
            DocumentId = id,
        };
        var result = await _mediator.Send(deleteDocumentTokenCommand);
        return Ok(result);
    }

    /// <summary>
    /// Read text Document
    /// </summary>
    /// <param name="id"></param>
    /// <param name="isVersion"></param>
    /// <returns></returns>
    [HttpGet("Document/{id}/readText")]
    public async Task<IActionResult> ReadTextDocument(Guid id, bool isVersion)
    {
        var commnad = new DownloadDocumentCommand
        {
            Id = id,
            IsVersion = isVersion
        };
        var downloadDocumentResponse = await _mediator.Send(commnad);

        if (!downloadDocumentResponse.Success)
        {
            return GenerateResponse(downloadDocumentResponse);
        }

        var downloadDocument = downloadDocumentResponse.Data;
        string utfString = Encoding.UTF8.GetString(downloadDocument.Data, 0, downloadDocument.Data.Length);
        return Ok(new { result = new string[] { utfString } });

    }

    /// <summary>
    /// Get Document Meta Tags
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("Document/{id}/getMetatag")]
    public async Task<IActionResult> GetDocumentMetatags(Guid id)
    {
        var commnad = new GetDocumentMetaDataByIdQuery
        {
            DocumentId = id,
        };
        var documentMetas = await _mediator.Send(commnad);
        return Ok(documentMetas);
    }

    /// <summary>
    /// Check Document By User
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("Document/{id}/userSignBy")]
    public async Task<IActionResult> CheckDocumentByUser(Guid id)
    {
        var commnad = new DocumentSignByUserCommand
        {
            DocumentId = id,
        };
        var flag = await _mediator.Send(commnad);
        return Ok(flag);
    }

    /// <summary>
    /// Check Document Permission
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("Document/{id}/permission")]
    public async Task<IActionResult> CheckDocumentPermission(Guid id)
    {
        var commnad = new CheckDocumentPermissionCommand
        {
            DocumentId = id,
        };
        var result = await _mediator.Send(commnad);
        return Ok(result.Data);
    }

    [HttpPost("Document/ArchiveRetentionPeriod")]
    // [ClaimCheck("Manage_Archive_Retention_Period")]
    public async Task<IActionResult> ArchiveRetentionPeriod(AddUpdateArchiveRetentionCommand addUpdateArchiveRetentionCommand)
    {
        var flag = await _mediator.Send(addUpdateArchiveRetentionCommand);
        return Ok(flag);
    }
}
