﻿using System;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.MediatR;

namespace QualityManagement.API.Controllers.DocumentVersion;

/// <summary>
/// Document
/// </summary>
[Route("api/[controller]")]
[ApiController]
[Authorize]
public class DocumentVersionController(IMediator _mediator) : BaseController
{
    /// <summary>
    /// Create a document.
    /// </summary>
    /// <param name="uploadNewDocumentVersionCommand"></param>
    /// <returns></returns>
    [HttpPost]
    [DisableRequestSizeLimit]
    [Produces("application/json", "application/xml", Type = typeof(DocumentVersionDto))]
    public async Task<IActionResult> AddDocumentVersion([FromForm] UploadNewDocumentVersionCommand uploadNewDocumentVersionCommand)
    {
        var result = await _mediator.Send(uploadNewDocumentVersionCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Add Document Chunk Version
    /// </summary>
    /// <param name="uploadNewDocumentChunkVersionCommand"></param>
    /// <returns></returns>
    [HttpPost("chunk")]
    [DisableRequestSizeLimit]
    [Produces("application/json", "application/xml", Type = typeof(DocumentVersionDto))]
    public async Task<IActionResult> AddDocumentChunkVersion(UploadNewDocumentChunkVersionCommand uploadNewDocumentChunkVersionCommand)
    {
        var result = await _mediator.Send(uploadNewDocumentChunkVersionCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Gets the document versions.
    /// </summary>
    /// <param name="id">The identifier.</param>
    /// <returns></returns>
    [HttpGet("{id}")]
    [ClaimCheck("ALL_VIEW_VERSION_HISTORY", "ASSIGNED_VIEW_VERSION_HISTORY")]
    public async Task<IActionResult> GetDocumentVersions(Guid id)
    {
        var updateUserProfilePhotoCommand = new GetDocumentVersionCommand()
        {
            Id = id
        };
        var result = await _mediator.Send(updateUserProfilePhotoCommand);
        return Ok(result);
    }
    /// <summary>
    /// Restores the document version.
    /// </summary>
    /// <param name="id">The identifier.</param>
    /// <param name="versionId">The version identifier.</param>
    /// <returns></returns>
    [HttpPost("{id}/restore/{versionId}")]
    [ClaimCheck("ALL_RESTORE_VERSION")]
    public async Task<IActionResult> RestoreDocumentVersion(Guid id, Guid versionId)
    {
        var updateUserProfilePhotoCommand = new RestoreDocumentVersionCommand()
        {
            Id = versionId,
            DocumentId = id
        };
        var result = await _mediator.Send(updateUserProfilePhotoCommand);
        return GenerateResponse(result);
    }
}
