﻿using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Resources;
using QualityManagement.MediatR;
using QualityManagement.MediatR.EmployeeCourses;
using QualityManagement.Repository;
using System;
using System.Linq;
using System.Threading.Tasks;

namespace QualityManagement.API.Controllers;
/// <summary>
/// Controller For Employee Course
/// </summary>
/// <returns></returns>
[Route("api/[controller]")]
[ApiController]
[Authorize]
public class EmployeeCourseController(IMediator mediator) : BaseController
{
    /// <summary>
    /// Assigns the course to company.
    /// </summary>
    /// <param name="assignCourseToCompanyCommand">Assign course to company.</param>
    /// <returns></returns>
    [HttpPost]
    [Produces("application/json", "application/xml", Type = typeof(CompanyProfileDto))]
    public async Task<IActionResult> AssignCourseToEmployee(AssignCourseToEmployeeCommand assignCourseToCompanyCommand)
    {
        var result = await mediator.Send(assignCourseToCompanyCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Search Employee for Company
    /// </summary>
    /// <param name="employeeResource"></param>
    /// <returns></returns>
    [HttpGet]
    [Produces("application/json", "application/xml", Type = typeof(EmployeeCourseList))]
    public async Task<IActionResult> GetEmployees([FromQuery] EmployeeResource employeeResource)
    {
        var getEmployeeListCommand = new SearchEmployeeListCommand
        {
            EmployeeResource = employeeResource
        };
        var result = await mediator.Send(getEmployeeListCommand);

        var paginationMetadata = new
        {
            totalCount = result.TotalCount,
            pageSize = result.PageSize,
            SkipStatusCodePagesAttribute = result.Skip,
            totalPages = result.TotalPages
        };
        Response.Headers.Append("X-Pagination",
            Newtonsoft.Json.JsonConvert.SerializeObject(paginationMetadata));

        return Ok(result);
    }

    /// <summary>
    /// Get assigned courses
    /// </summary>
    /// <returns></returns>
    [HttpGet("assignedCourses")]
    [Produces("application/json", "application/xml", Type = typeof(EmployeeCourseDto))]
    public async Task<IActionResult> GetAssignedCourses([FromQuery] AssignedEmployeeCourseResource assignedEmployeeCourseResource)
    {
        var command = new GetAssignedEmployeeCoursesCommand()
        {
            AssignedEmployeeCourseResource = assignedEmployeeCourseResource
        };
        var result = await mediator.Send(command);

        var paginationMetadata = new
        {
            totalCount = result.TotalCount,
            pageSize = result.PageSize,
            skip = result.Skip,
            totalPages = result.TotalPages
        };
        Response.Headers.Append("X-Pagination",
            Newtonsoft.Json.JsonConvert.SerializeObject(paginationMetadata));
        return Ok(result);
    }
    /// <summary>
    /// Get assigned courses by employeeId
    /// </summary>
    /// <returns></returns>
    [HttpGet("{id}")]
    [Produces("application/json", "application/xml", Type = typeof(EmployeeCourseDto))]
    public async Task<IActionResult> GetAssignedCourseByEmployeeId(Guid id)
    {
        var command = new GetAssignedEmployeeCoursesCommand()
        {
            AssignedEmployeeCourseResource = new AssignedEmployeeCourseResource
            {
                UserId = id,
                PageSize = 0,
            }
        };
        var result = await mediator.Send(command);
        return Ok(result);
    }

    /// <summary>
    /// Get Employee course seesion
    /// </summary>
    /// <param name="link"></param>
    /// <returns></returns>
    [AllowAnonymous]
    [HttpGet("link/{link}")]
    [Produces("application/json", "application/xml", Type = typeof(EmployeeCourseDetailDto))]
    public async Task<IActionResult> GetEmployeeCourse(string link)
    {
        var command = new GetEmployeeCourseByLinkCommand { Link = link };
        var result = await mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get Employees by course id.
    /// </summary>
    /// <returns></returns>
    [HttpGet("{courseId}/employees")]
    [Produces("application/json", "application/xml", Type = typeof(EmployeeCourseDto))]
    public async Task<IActionResult> GetEmployeeCourseByCourseId(Guid courseId, [FromQuery] AssignedEmployeeCourseResource assignedEmployeeCourseResource)
    { 
        assignedEmployeeCourseResource.CourseId = courseId;
        var command = new GetAssignedEmployeeCoursesCommand()
        {
            AssignedEmployeeCourseResource = assignedEmployeeCourseResource
        };
        var result = await mediator.Send(command);
        var paginationMetadata = new
        {
            totalCount = result.TotalCount,
            pageSize = result.PageSize,
            skip = result.Skip,
            totalPages = result.TotalPages
        };
        Response.Headers.Append("X-Pagination",
            Newtonsoft.Json.JsonConvert.SerializeObject(paginationMetadata));
        return Ok(result);
    }

    /// <summary>
    /// Get Employee All Asigned pending Cources
    /// </summary>
    /// <returns></returns>
    [HttpGet("my-courses/pending")]
    public async Task<IActionResult> GetEmployeeAllAsignedPendingCources()
    {
        var result = await mediator.Send(new GetEmployeeAsignAllPendingCoursesCommand());
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get Employee All Asigned In-Progress Cources
    /// </summary>
    /// <returns></returns>
    [HttpGet("my-courses/in-progress")]
    public async Task<IActionResult> GetEmployeeAllAsignedInProgressCources()
    {
        var result = await mediator.Send(new GetEmployeeAsignAllInProgressCoursesCommand());
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get Employee All Asigned In-Progress Cources
    /// </summary>
    /// <returns></returns>
    [HttpGet("my-courses/completed")]
    public async Task<IActionResult> GetEmployeeAllAsignedcompletedCources()
    {
        var result = await mediator.Send(new GetEmployeeAsignAllCompleteCoursesCommand());
        return GenerateResponse(result);
    }

    /// <summary>
    /// download certificate by employee course id
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}/download-certificate")]
    //[Produces("application/json", "application/xml", Type = typeof(EmployeeCourseDto))]
    public async Task<IActionResult> DownloadCourseCertificate(Guid id)
    {
        var command = new DownloadEmployeeCourseCertificateCommand()
        {
            Id = id
        };

        var result = await mediator.Send(command);
        if (!result.Success)
        {
            return GenerateResponse(result);
        }
        var downloadDocument = result.Data;
        return File(downloadDocument.Data, downloadDocument.ContentType, downloadDocument.FileName);
    }
    /// <summary>
    /// Get Employee All Asigned Course Status Counts
    /// </summary>
    /// <returns></returns>
    [HttpGet("{courseId}/counts")]
    public async Task<IActionResult> GetAllAsignEmployeeCourseStatusCount(Guid courseId)
    {
        var result = await mediator.Send(new GetAssignedEmployeeCoursesStatusCountCommand() { CourseId = courseId });
        return GenerateResponse(result);
    }

}
