﻿using System;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Dto.Document;
using QualityManagement.MediatR;
using QualityManagement.MediatR.Capa;

namespace QualityManagement.API.Controllers;

/// <summary>
/// Controller for NonConformanceResponse
/// </summary>
[Route("api/[controller]")]
[ApiController]
[Authorize]
public class NonConformanceResponseController(IMediator mediator) : BaseController
{
    /// <summary>
    /// Create A NonConformanceResponse
    /// </summary>
    /// <param name="command"></param>
    /// <returns></returns>
    [HttpPost]
    [ClaimCheck("ADD_NON_CONFORMANCE_RESPONSE")]
    public async Task<IActionResult> AddNonConformanceResponse(AddNonConformanceResponseCommand command)
    {
        var result = await mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Create A NonConformanceResponseAttachment
    /// </summary>
    /// <param name="command"></param>
    /// <returns></returns>
    [HttpPost("Attachment")]
    public async Task<IActionResult> AddNonConformanceResponseAttachment(AddNonConformanceRespAttachmentCommand command)
    {
        var result = await mediator.Send(command);
        return GenerateResponse(result);
    }
    /// <summary>
    /// Upload A NonConformanceResponseAttachment Chunk
    /// </summary>
    /// <param name="command"></param>
    /// <returns></returns>
    [HttpPost("Attachment/Chunk")]
    public async Task<IActionResult> UploadNonConformanceResponseAttachmentChunk([FromForm] UploadNonConformanceRespAttachmenttChunkCommand command)
    {
        var result = await mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Mark all chunks of a Attachment  as uploaded.
    /// </summary>
    /// <param name="markAttachmentChunksUploadedCommand"></param>
    /// <returns></returns>
    [HttpPost("chunk/uploadStatus")]
    [Produces("application/json", "application/xml", Type = typeof(NonConformanceRespAttachmentDto))]
    public async Task<IActionResult> MarkSessionMediaAllChunkUpload(MarkAttachmentChunksUploadedCommand markAttachmentChunksUploadedCommand)
    {
        var result = await mediator.Send(markAttachmentChunksUploadedCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Delete NCResponse Attachment
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpDelete("Attachment/{id}")]
    public async Task<IActionResult> DeleteNcResponseAttachment(Guid id)
    {
        var result = await mediator.Send(new DeleteNonConformanceResponseAttachmentCommand() { NonConformanceRespAttachmentId = id });
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get Download NcAttachment chunk.
    /// </summary>
    /// <param name="NcRespAttachmentId"></param>
    /// <param name="chunkIndex"></param>
    /// <returns></returns>
    [HttpGet("{NcRespAttachmentId}/chunk/{chunkIndex}/download")]
    public async Task<IActionResult> DownloadNcRespAttachment(Guid NcRespAttachmentId, int chunkIndex)
    {
        var result = new DownloadNonConformanceAttachmentCommand
        {
            NcRespAttachmentId = NcRespAttachmentId,
            ChunkIndex = chunkIndex
        };

        var downloadAttachmentResponse = await mediator.Send(result);
        if (!downloadAttachmentResponse.Success)
        {
            return GenerateResponse(downloadAttachmentResponse);
        }
        return Ok(downloadAttachmentResponse.Data);
    }


    /// <summary>
    /// Update NonConformanceResponse
    /// </summary>
    /// <param name="id"></param>
    /// <param name="command"></param>
    /// <returns></returns>
    [HttpPut("{id}")]
    public async Task<IActionResult> UpdateNonConformanceResponse(Guid id, UpdateNonConformanceResponseCommand command)
    {
        command.Id = id;
        var result = await mediator.Send(command);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Delete NonConformance Response by id
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpDelete("{id}")]
    public async Task<IActionResult> DeleteNonConformanceResponse(Guid id)
    {
        var result = await mediator.Send(new DeleteNonConformanceResponseCommand() { Id = id });
        return GenerateResponse(result);
    }

    // <summary>
    /// Get NonConformance Response by Id
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>
    [HttpGet("{id}")]
    public async Task<IActionResult> GetNonConformanceResponse(Guid id)
    {
        var result = await mediator.Send(new GetNonConformanceResponseCommand() { Id = id });
        return GenerateResponse(result);
    }

    /// <summary>
    /// Gets the document token.
    /// </summary>
    /// <param name="id">The identifier.</param>
    /// <returns></returns>
    [HttpGet("{id}/token")]
    [AllowAnonymous]
    public async Task<IActionResult> GetDocumentToken(Guid id)
    {
        var result = await mediator.Send(new GetDocumentTokenByDocumentIdCommand { Id = id });
        return Ok(new { result = result.Data });
    }

    /// <summary>
    /// Get Document File By Token
    /// </summary>
    /// <param name="id"></param>
    /// <param name="officeViewerRequest"></param>
    /// <returns></returns>
    [HttpGet("{id}/officeviewer")]
    [AllowAnonymous]
    public async Task<IActionResult> GetDocumentFileByToken(Guid id, [FromQuery] OfficeViewerRequest officeViewerRequest)

    {
        var command = new DownloadNonConformanceRespAttachmenttChunkCommand
        {
            Id = id,
            Token = officeViewerRequest.Token
        };

        var downloadDocumentResponse = await mediator.Send(command);

        if (!downloadDocumentResponse.Success)
        {
            return GenerateResponse(downloadDocumentResponse);
        }

        var downloadDocument = downloadDocumentResponse.Data;
        return File(downloadDocument.Data, downloadDocument.ContentType, downloadDocument.FileName);
    }
}
