﻿using System;
using System.Linq;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Resources;
using QualityManagement.MediatR;
using QualityManagement.Repository;

namespace QualityManagement.API.Controllers;
/// <summary>
/// Controller for Supplier
/// </summary>
/// <returns></returns>
[Route("api/[controller]")]
[ApiController]
[Authorize]
public class SupplierController(IMediator mediator) : BaseController
{
    /// <summary>
    /// Create A Supplier
    /// </summary>
    /// <param name="addSupplierCommand"></param>
    /// <returns></returns>
    [HttpPost]
    [Produces("application/json", "application/xml", Type = typeof(SupplierDto))]
    [ClaimCheck("ADD_SUPPLIER")]
    public async Task<IActionResult> CreateSupplier(AddSupplierCommand addSupplierCommand)
    {
        var result = await mediator.Send(addSupplierCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Update Supplier
    /// </summary>
    /// <param name="id"></param>
    /// <param name="updateSupplierCommand"></param>
    /// <returns></returns>

    [HttpPut("{id}")]
    [Produces("application/json", "application/xml", Type = typeof(SupplierDto))]
    [ClaimCheck("EDIT_SUPPLIER")]
    public async Task<IActionResult> UpdateSupplier(Guid id, UpdateSupplierCommand updateSupplierCommand)
    {
        updateSupplierCommand.Id = id;
        var result = await mediator.Send(updateSupplierCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get Supplier By id.
    /// </summary>
    /// <param name="id"></param>
    /// <returns></returns>

    [HttpGet("{id}")]
    [Produces("application/json", "application/xml", Type = typeof(SupplierDto))]
    public async Task<IActionResult> GetSupplier(Guid id)
    {
        var getSupplierCommand = new GetSupplierCommand { Id = id };
        var result = await mediator.Send(getSupplierCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Get All Suppliers
    /// </summary>
    /// <param name="supplierResource"></param>
    /// <returns></returns>
    [HttpGet]
    [Produces("application/json", "application/xml", Type = typeof(SupplierList))]
    [ClaimCheck("VIEW_SUPPLIER")]
    public async Task<IActionResult> GetSuppliers([FromQuery] SupplierResource supplierResource)
    {
        var getAllSupplierCommand = new GetAllSupplierCommand
        {
            SupplierResource = supplierResource
        };
        var result = await mediator.Send(getAllSupplierCommand);

        var paginationMetadata = new
        {
            totalCount = result.TotalCount,
            pageSize = result.PageSize,
            skip = result.Skip,
            totalPages = result.TotalPages
        };
        Response.Headers.Append("X-Pagination",
            Newtonsoft.Json.JsonConvert.SerializeObject(paginationMetadata));
        return Ok(result);
    }

    /// <summary>
    /// Delete Supplier By Id
    /// </summary>
    /// <param name="Id"></param>
    /// <returns></returns>
    [HttpDelete("{Id}")]
    [ClaimCheck("DELETE_SUPPLIER")]
    public async Task<IActionResult> DeleteSupplier(Guid Id)
    {
        var deleteSupplierCommand = new DeleteSupplierCommand { Id = Id };
        var result = await mediator.Send(deleteSupplierCommand);
        return GenerateResponse(result);
    }

    /// <summary>
    /// Gets the new SupplierCode.
    /// </summary>
    /// <returns></returns>
    [HttpGet("GetNewSupplierCode")]
    public async Task<IActionResult> GetNewSupplierCode()
    {
        var getNewNCNumberQuery = new GetNewSupplierCodeCommand();
        var response = await mediator.Send(getNewNCNumberQuery);
        return Ok(new
        {
            supplierCode = response
        });
    }
}

