﻿using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto;
using QualityManagement.MediatR;
using System;
using System.Collections.Generic;
using System.Threading.Tasks;

namespace QualityManagement.API.Controllers.Workflow
{
    /// <summary>
    /// Controller  For WorkFlowTransition
    /// </summary>
    /// <returns></returns>
    [Route("api/[controller]")]
    [ApiController]
    [Authorize]
    public class WorkflowTransitionController(IMediator _mediator) : BaseController
    {
        /// <summary>
        /// Creates the workflows transition.
        /// </summary>
        /// <param name="command">The command.</param>
        /// <returns></returns>
        [HttpPost]
        public async Task<IActionResult> CreateWorkflowTransition(AddWorkflowTransitionCommand command)
        {
            var result = await _mediator.Send(command);
            return GenerateResponse(result);

        }

        /// <summary>
        /// Updates the workflow transition.
        /// </summary>
        /// <param name="command">The update wrokflow setting command.</param>
        /// <returns></returns>
        [HttpPut]
        [Produces("application/json", "application/xml", Type = typeof(WorkflowStepDto))]
        public async Task<IActionResult> UpdateWorkflowTransition(UpdateWorkflowTransitionCommand command)
        {
            var result = await _mediator.Send(command);
            return GenerateResponse(result);
        }

        /// <summary>
        /// Delete Workflow Transition
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        [HttpDelete("{id}")]
        public async Task<IActionResult> DeleteWorkflowTransition(Guid id)
        {
            var query = new DeleteWorkflowTransitionCommand { Id = id };
            var result = await _mediator.Send(query);
            return GenerateResponse(result);
        }

        /// <summary>
        /// Gets the workflow step.
        /// </summary>
        /// <param name="id">The identifier.</param>
        /// <returns></returns>
        [HttpGet("{id}")]
        public async Task<IActionResult> GetWorkflowTransition(Guid id)
        {
            var query = new GetWorkflowTransitionQuery { Id = id };
            var result = await _mediator.Send(query);
            return Ok(result.Data);
        }

        /// <summary>
        /// Get all workflow transition
        /// </summary>
        /// <returns></returns>
        [HttpGet(Name = "GetWorkflowTransitions ")]
        [Produces("application/json", "application/xml", Type = typeof(List<WorkflowTransitionDto>))]
        public async Task<IActionResult> GetWorkflowTransitions()
        {
            var query = new GetAllWorkflowTransitionQuery { };
            var result = await _mediator.Send(query);
            return Ok(result.Data);
        }

        /// <summary>
        ///  Perform Next Transition
        /// </summary>
        /// <param name="command">The command.</param>
        /// <returns></returns>
        [HttpPost("nextTransition")]
        [Produces("application/json", "application/xml", Type = typeof(PerformWorkflowTransitionToNextTransitionCommand))]
        public async Task<IActionResult> PerformNextTransition(PerformWorkflowTransitionToNextTransitionCommand command)
        {
            var result = await _mediator.Send(command);
            return GenerateResponse(result);

        }

        /// <summary>
        /// Perform Workflow Transition With Signature And Document
        /// </summary>
        /// <param name="command">The command.</param>
        /// <returns></returns>
        [HttpPost("nextTransitionWithDocumentAndSignature")]
        [Produces("application/json", "application/xml", Type = typeof(PerformWorkflowTransitionWithSignatureAndDocumentCommand))]
        public async Task<IActionResult> PerformWorkflowTransitionWithSignatureAndDocumentCommand(PerformWorkflowTransitionWithSignatureAndDocumentCommand command)
        {
            var result = await _mediator.Send(command);
            return GenerateResponse(result);

        }

    }
}
