﻿using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using System.Collections.Generic;
using System.Linq;

namespace QualityManagement.API.Helpers;

using Microsoft.OpenApi.Models;
using Swashbuckle.AspNetCore.SwaggerGen;
/// <summary>
/// FileUploadOperationFilter is an implementation of IOperationFilter that modifies the OpenAPI operation to support file uploads.
/// </summary>
public class FileUploadOperationFilter : IOperationFilter
{
    /// <summary>
    /// Modifies the OpenAPI operation to support file uploads by adding a multipart/form-data request body if the action method has parameters decorated with FromFormAttribute.
    /// </summary>
    /// <param name="operation"></param>
    /// <param name="context"></param>
    public void Apply(OpenApiOperation operation, OperationFilterContext context)
    {
        var formParameters = context.MethodInfo.GetParameters()
            .Where(p => p.GetCustomAttributes(typeof(FromFormAttribute), false).Any())
            .SelectMany(p => p.ParameterType.GetProperties());

        if (formParameters.Any())
        {
            operation.RequestBody = new OpenApiRequestBody
            {
                Content = new Dictionary<string, OpenApiMediaType>
                {
                    ["multipart/form-data"] = new OpenApiMediaType
                    {
                        Schema = new OpenApiSchema
                        {
                            Type = "object",
                            Properties = formParameters.ToDictionary(
                                prop => prop.Name,
                                prop =>
                                {
                                    if (prop.PropertyType == typeof(IFormFile))
                                    {
                                        return new OpenApiSchema
                                        {
                                            Type = "string",
                                            Format = "binary"
                                        };
                                    }

                                    return new OpenApiSchema { Type = "string" };
                                }
                            )
                        }
                    }
                }
            };
        }
    }
}
