﻿using System;
using Microsoft.AspNetCore.Identity.EntityFrameworkCore;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;


namespace QualityManagement.Domain;

public class QMSDbContext : IdentityDbContext<User, Role, Guid, UserClaim, UserRole, UserLogin, RoleClaim, UserToken>
{
    public QMSDbContext(DbContextOptions options) : base(options)
    {
    }
    public override DbSet<User> Users { get; set; }
    public override DbSet<Role> Roles { get; set; }
    public override DbSet<UserClaim> UserClaims { get; set; }
    public override DbSet<UserRole> UserRoles { get; set; }
    public override DbSet<UserLogin> UserLogins { get; set; }
    public override DbSet<RoleClaim> RoleClaims { get; set; }
    public override DbSet<UserToken> UserTokens { get; set; }
    public DbSet<NLog> NLog { get; set; }
    public DbSet<Document> Documents { get; set; }
    public DbSet<Category> Categories { get; set; }
    public virtual DbSet<DocumentRolePermission> DocumentRolePermissions { get; set; }
    public virtual DbSet<DocumentUserPermission> DocumentUserPermissions { get; set; }
    public DbSet<DocumentAuditTrail> DocumentAuditTrails { get; set; }
    public DbSet<UserNotification> UserNotifications { get; set; }
    public DbSet<LoginAudit> LoginAudits { get; set; }
    public DbSet<DocumentToken> DocumentTokens { get; set; }
    public DbSet<Reminder> Reminders { get; set; }
    public DbSet<ReminderNotification> ReminderNotifications { get; set; }
    public DbSet<ReminderUser> ReminderUsers { get; set; }
    public DbSet<ReminderScheduler> ReminderSchedulers { get; set; }
    public DbSet<HalfYearlyReminder> HalfYearlyReminders { get; set; }
    public DbSet<QuarterlyReminder> QuarterlyReminders { get; set; }
    public DbSet<DailyReminder> DailyReminders { get; set; }
    public DbSet<EmailSMTPSetting> EmailSMTPSettings { get; set; }
    public DbSet<SendEmail> SendEmails { get; set; }
    public DbSet<DocumentComment> DocumentComments { get; set; }
    public DbSet<DocumentVersion> DocumentVersions { get; set; }
    public DbSet<DocumentMetaData> DocumentMetaDatas { get; set; }
    public DbSet<DocumentShareableLink> DocumentShareableLinks { get; set; }
    public DbSet<StorageSetting> StorageSettings { get; set; }
    public DbSet<DocumentStatus> DocumentStatuses { get; set; }
    public DbSet<CompanyProfile> CompanyProfiles { get; set; }
    public DbSet<PageHelper> PageHelpers { get; set; }
    public DbSet<DocumentIndex> DocumentIndexes { get; set; }
    public DbSet<Workflow> Workflows { get; set; }
    public DbSet<WorkflowStep> WorkflowSteps { get; set; }
    public DbSet<WorkflowInstance> WorkflowInstances { get; set; }
    public DbSet<WorkflowStepInstance> WorkflowStepInstances { get; set; }
    public DbSet<WorkflowTransition> WorkflowTransitions { get; set; }
    public DbSet<WorkflowTransitionInstance> WorkflowTransitionInstances { get; set; }
    public DbSet<PendingTransition> PendingTransitions { get; private set; }
    public DbSet<AllowFileExtension> AllowFileExtensions { get; set; }
    public DbSet<WorkflowInstanceEmailSender> WorkflowInstanceEmailSenders { get; set; }
    public DbSet<DocumentSignature> DocumentSignatures { get; set; }
    public DbSet<CustomCategory> CustomCategories { get; private set; }
    public DbSet<FileRequest> FileRequests { get; set; }
    public DbSet<FileRequestDocument> FileRequestDocuments { get; set; }
    public DbSet<Client> Clients { get; set; }
    public DbSet<MatTableSetting> MatTableSettings { get; set; }
    public DbSet<DocumentChunk> DocumentChunks { get; set; }
    public DbSet<CategoryUserPermission> CategoryUserPermissions { get; set; }
    public DbSet<CategoryRolePermission> CategoryRolePermissions { get; set; }
    public DbSet<DocumentMetaTag> DocumentMetaTags { get; set; }
    public DbSet<UserOpenaiMsg> UserOpenaiMsgs { get; set; }
    public DbSet<AIPromptTemplate> AIPromptTemplates { get; set; }
    public DbSet<WorkflowTransitionRole> WorkflowTransitionRoles { get; set; }
    public DbSet<WorkflowTransitionUser> WorkflowTransitionUsers { get; set; }
    public DbSet<Data.Action> Actions { get; set; }
    public DbSet<Page> Pages { get; set; }
    public DbSet<EmailTemplate> EmailTemplates { get; set; }
    public DbSet<Course> Courses { get; set; }
    public DbSet<CourseSession> CourseSessions { get; set; }
    public DbSet<CourseSessionMedia> CourseSessionMedias { get; set; }
    public DbSet<CourseSessionQuiz> CourseSessionQuizzes { get; set; }
    public DbSet<QuizQuestion> QuizQuestions { get; set; }
    public DbSet<QuizQuestionOption> QuizQuestionOptions { get; set; }
    public DbSet<EmployeeCourse> EmployeeCourses { get; set; }
    public DbSet<CourseSessionMediaChunk> CourseSessionMediaChunks { get; set; }
    public DbSet<CourseSessionMediaToken> CourseSessionMediaTokens { get; set; }
    public DbSet<EmployeeCourseSession> EmployeeCourseSessions { get; set; }
    public DbSet<EmployeeCourseSessionMedia> EmployeeCourseSessionMedias { get; set; }
    public DbSet<EmployeeCourseSessionQuiz> EmployeeCourseSessionQuizzes { get; set; }
    public DbSet<EmployeeCourseSessionQuizAnswer> EmployeeCourseSessionQuizAnswers { get; set; }
    public DbSet<EmployeeCourseSessionQuizAnswerOption> EmployeeCourseSessionQuizAnswerOptions { get; set; }
    //public DbSet<EmployeeCourseSessionEmail> EmployeeCourseSessionEmails { get; set; }
    public DbSet<AuditTemplate> AuditTemplates { get; set; }
    public DbSet<AuditTemplateQuestion> AuditTemplateQuestions { get; set; }
    public DbSet<Audit> Audits { get; set; }
    public DbSet<AuditResponse> AuditResponses { get; set; }
    public DbSet<NonConformance> NonConformances { get; set; }
    public DbSet<NonConformanceRespAttachment> NonConformanceRespAttachments { get; set; }
    public DbSet<NonConformanceResponse> NonConformanceResponses { get; set; }
    public DbSet<NonConformanceResponseType> NonConformanceResponseTypes { get; set; }
    public DbSet<CapaAction> CapaActions { get; set; }
    public DbSet<CapaAttachment> CapaAttachments { get; set; }
    public DbSet<CapaRequest> CapaRequests { get; set; }
    public DbSet<CapaReview> CapaReviews { get; set; }
    public DbSet<CapaRootCause> CapaRootCauses { get; set; }
    public DbSet<CapaRootCauseMethodology> CapaRootCauseMethodologies { get; set; }
    public DbSet<NonConformanceRespAttachmentChunk> NonConformanceRespAttachmentChunks { get; set; }
    public DbSet<CapaAttachmentChunk> CapaAttachmentChunks { get; set; }
    public DbSet<QuestionOption> QuestionOptions { get; set; }
    public DbSet<Risk> Risks { get; set; }
    public DbSet<RiskReview> RiskReviews { get; set; }
    public DbSet<RiskCategory> RiskCategories { get; set; }
    public DbSet<RiskControl> RiskControls { get; set; }
    public DbSet<RiskMatrixLevel> RiskMatrixLevels { get; set; }
    public DbSet<RiskAttachment> RiskAttachments { get; set; }
    public DbSet<RiskAttachmentChunk> RiskAttachmentChunks { get; set; }
    public DbSet<Complaint> Complaints { get; set; }
    public DbSet<ComplaintAction> ComplaintActions { get; set; }
    public DbSet<ComplaintAttachment> ComplaintAttachments { get; set; }
    public DbSet<ComplaintAttachmentChunk> ComplaintAttachmentChunks { get; set; }
    public DbSet<ComplaintComment> ComplaintComments { get; set; }
    public DbSet<ComplaintInvestigation> ComplaintInvestigations { get; set; }
    public DbSet<ComplaintType> ComplaintTypes { get; set; }
    public DbSet<ComplaintRootCause> ComplaintRootCauses { get; set; }
    public DbSet<Supplier> Suppliers { get; set; }
    public DbSet<SupplierAudit> SupplierAudits { get; set; }
    public DbSet<SupplierCertification> SupplierCertifications { get; set; }
    public DbSet<SupplierContact> SupplierContacts { get; set; }
    public DbSet<SupplierPerformance> SupplierPerformances { get; set; }
    public DbSet<SupplierQualification> SupplierQualifications { get; set; }
    public DbSet<AuditResponseLog> AuditResponseLogs { get; set; }
    public DbSet<CapaRequestLog> CapaRequestLogs { get; set; }
    public DbSet<ComplaintLog> ComplaintLogs { get; set; }
    public DbSet<NonConformanceLog> NonConformanceLogs { get; set; }
    public DbSet<RiskLog> RiskLogs { get; set; }
    public DbSet<AuditReviewer> AuditReviewers { get; set; }
    public DbSet<AuditReviewComment> AuditReviewComments { get; set; }
    public DbSet<Module> Modules { get; set; }
    public DbSet<ArchiveRetention> ArchiveRetentions { get; set; }

    protected override void OnModelCreating(ModelBuilder builder)
    {

        base.OnModelCreating(builder);

        builder.Entity<Document>(entity =>
        {
            if (Database.IsSqlServer()) // For SQL Server
            {
                entity.Property(e => e.IV).HasColumnType("VARBINARY(MAX)").IsRequired(false);
                entity.Property(e => e.Key).HasColumnType("VARBINARY(MAX)").IsRequired(false);


            }
            else //For MySQL
            {
                entity.Property(e => e.IV).HasColumnType("LONGBLOB").IsRequired(false);  // or BLOB
                entity.Property(e => e.Key).HasColumnType("LONGBLOB").IsRequired(false);
            }

        });
        builder.Entity<Document>()
            .HasIndex(b => new { b.Name, b.CategoryId, b.IsArchive, b.IsDeleted })
            .IsUnique();

        builder.Entity<DocumentVersion>(entity =>
        {
            if (Database.IsSqlServer()) // For SQL Server
            {
                entity.Property(e => e.IV).HasColumnType("VARBINARY(MAX)").IsRequired(false);
                entity.Property(e => e.Key).HasColumnType("VARBINARY(MAX)").IsRequired(false);
            }
            else //For MySQL
            {
                entity.Property(e => e.IV).HasColumnType("LONGBLOB").IsRequired(false);  // or BLOB
                entity.Property(e => e.Key).HasColumnType("LONGBLOB").IsRequired(false);
            }

        });
        builder.Entity<DocumentVersion>().HasIndex(b => new { b.DocumentId, b.Url, b.CreatedBy });

        builder.Entity<Category>(entity =>
        {
            entity.HasOne(x => x.Parent)
                .WithMany(x => x.Children)
                .HasForeignKey(x => x.ParentId)
                .IsRequired(false)
                .OnDelete(DeleteBehavior.Restrict);

            entity.HasOne(c => c.CreatedByUser)
                .WithMany()
                .HasForeignKey(c => c.CreatedBy)
                .OnDelete(DeleteBehavior.ClientSetNull);
        });
        builder.Entity<Category>()
            .HasIndex(b => new { b.Name, b.ParentId, b.IsArchive, b.IsDeleted });


        builder.Entity<WorkflowInstance>(entity =>
        {
            entity.HasOne(x => x.Document)
                .WithMany(x => x.WorkflowInstances)
                .HasForeignKey(x => x.DocumentId)
                .IsRequired(false)
                .OnDelete(DeleteBehavior.Restrict);
        });

        //builder.Entity<Document>(entity =>
        //{
        //    entity.HasMany(x => x.WorkflowInstances)
        //        .WithOne(x => x.Document)
        //        .HasForeignKey(x => x.DocumentId)
        //        .IsRequired(false)
        //        .OnDelete(DeleteBehavior.Restrict);
        //});

        builder.Entity<UserClaim>()
            .HasOne(uc => uc.Action)
            .WithMany()
            .HasForeignKey(uc => uc.ActionId)
            .OnDelete(DeleteBehavior.Restrict); // or NoAction

        builder.Entity<UserClaim>()
            .HasOne(uc => uc.Page)
            .WithMany()
            .HasForeignKey(uc => uc.PageId)
            .OnDelete(DeleteBehavior.Restrict); // or NoAction

        builder.Entity<UserClaim>()
            .HasOne(uc => uc.User)
            .WithMany()
            .HasForeignKey(uc => uc.UserId)
            .OnDelete(DeleteBehavior.Restrict); // or NoAction

        builder.Entity<User>(b =>
        {
            // Each User can have many UserClaims
            b.HasMany(e => e.UserClaims)
                .WithOne(e => e.User)
                .HasForeignKey(uc => uc.UserId)
                .IsRequired();

            // Each User can have many UserLogins
            b.HasMany(e => e.UserLogins)
                .WithOne(e => e.User)
                .HasForeignKey(ul => ul.UserId)
                .IsRequired();

            // Each User can have many UserTokens
            b.HasMany(e => e.UserTokens)
                .WithOne(e => e.User)
                .HasForeignKey(ut => ut.UserId)
                .IsRequired();

            // Each User can have many entries in the UserRole join table
            b.HasMany(e => e.UserRoles)
                .WithOne(e => e.User)
                .HasForeignKey(ur => ur.UserId)
                .IsRequired();
        });

        builder.Entity<Role>(b =>
        {
            // Each Role can have many entries in the UserRole join table
            b.HasMany(e => e.UserRoles)
                .WithOne(e => e.Role)
                .HasForeignKey(ur => ur.RoleId)
                .IsRequired();

            // Each Role can have many associated RoleClaims
            b.HasMany(e => e.RoleClaims)
                .WithOne(e => e.Role)
                .HasForeignKey(rc => rc.RoleId)
                .IsRequired();
        });

        builder.Entity<DocumentRolePermission>(entity =>
        {
            entity.HasOne(d => d.Document)
                .WithMany(p => p.DocumentRolePermissions)
                .HasForeignKey(d => d.DocumentId)
                .OnDelete(DeleteBehavior.ClientSetNull);

            entity.HasOne(d => d.Role)
                .WithMany(p => p.DocumentRolePermissions)
                .HasForeignKey(d => d.RoleId)
                .OnDelete(DeleteBehavior.ClientSetNull);
        });

        builder.Entity<DocumentUserPermission>(entity =>
        {

            entity.HasOne(d => d.Document)
                .WithMany(p => p.DocumentUserPermissions)
                .HasForeignKey(d => d.DocumentId)
                .OnDelete(DeleteBehavior.ClientSetNull);

            entity.HasOne(d => d.User)
                .WithMany(p => p.DocumentUserPermissions)
                .HasForeignKey(d => d.UserId)
                .OnDelete(DeleteBehavior.ClientSetNull);
        });

        builder.Entity<DocumentAuditTrail>(entity =>
        {
            entity.HasOne(d => d.Document)
              .WithMany(p => p.DocumentAuditTrails)
              .HasForeignKey(d => d.DocumentId)
              .OnDelete(DeleteBehavior.ClientSetNull);
        });
        builder.Entity<UserNotification>(entity =>
        {
            entity.HasOne(d => d.User)
                .WithMany(p => p.UserNotifications)
                .HasForeignKey(d => d.UserId)
                .OnDelete(DeleteBehavior.NoAction);

            entity.HasOne(d => d.Document)
                .WithMany(p => p.UserNotifications)
                .HasForeignKey(d => d.DocumentId)
                .OnDelete(DeleteBehavior.ClientSetNull);

        });

        builder.Entity<ReminderUser>(b =>
        {
            b.HasKey(e => new { e.ReminderId, e.UserId });
            b.HasOne(e => e.User)
              .WithMany()
              .HasForeignKey(ur => ur.UserId)
              .OnDelete(DeleteBehavior.NoAction);
        });

        builder.Entity<Document>()
         .HasQueryFilter(p => !p.IsDeleted);
        //.HasIndex(b => b.Url);

        builder.Entity<DocumentComment>(b =>
        {
            b.HasOne(e => e.CreatedByUser)
                .WithMany()
                .HasForeignKey(ur => ur.CreatedBy)
                .OnDelete(DeleteBehavior.Restrict);
        });

        builder.Entity<DocumentVersion>(b =>
        {
            b.HasOne(e => e.CreatedByUser)
                .WithMany()
                .HasForeignKey(ur => ur.CreatedBy)
                .OnDelete(DeleteBehavior.NoAction);
        });

        builder.Entity<Workflow>(entity =>
        {
            entity.HasOne(w => w.User)
                .WithMany()
                .HasForeignKey(w => w.UserId)
                .OnDelete(DeleteBehavior.Restrict);
        });

        builder.Entity<WorkflowStep>()
          .HasOne(wsi => wsi.Workflow)
          .WithMany(wi => wi.WorkflowSteps)
          .HasForeignKey(wsi => wsi.WorkflowId)
          .OnDelete(DeleteBehavior.Restrict);

        builder.Entity<WorkflowTransition>()
        .HasMany(wsi => wsi.WorkflowTransitionRoles)
        .WithOne(wi => wi.WorkflowTransition)
        .OnDelete(DeleteBehavior.Cascade);



        builder.Entity<WorkflowTransition>()
        .HasMany(wsi => wsi.WorkflowTransitionUsers)
        .WithOne(wi => wi.WorkflowTransition)
        .OnDelete(DeleteBehavior.Cascade);


        builder.Entity<WorkflowInstance>()
            .HasOne(wsi => wsi.Workflow)
            .WithMany(wi => wi.WorkflowInstances)
            .HasForeignKey(wsi => wsi.WorkflowId)
            .OnDelete(DeleteBehavior.NoAction);

        builder.Entity<WorkflowStepInstance>()
            .HasOne(wsi => wsi.WorkflowStep)
            .WithMany(wi => wi.WorkflowStepInstances)
            .HasForeignKey(wsi => wsi.StepId)
            .OnDelete(DeleteBehavior.NoAction);

        builder.Entity<WorkflowStepInstance>()
            .HasOne(wsi => wsi.WorkflowInstance)
            .WithMany(wi => wi.WorkflowStepInstances)
            .HasForeignKey(wsi => wsi.WorkflowInstanceId)
            .OnDelete(DeleteBehavior.NoAction);

        builder.Entity<WorkflowStepInstance>()
           .HasOne(wsi => wsi.User)
           .WithMany()
           .HasForeignKey(wsi => wsi.UserId)
           .OnDelete(DeleteBehavior.NoAction);

        builder.Entity<WorkflowTransition>(entity =>
        {
            // Foreign key: FromStepId
            entity.HasOne(wt => wt.FromWorkflowStep)
                  .WithMany(ws => ws.FromWorkflowTransitions)
                  .HasForeignKey(wt => wt.FromStepId)
                  .OnDelete(DeleteBehavior.Restrict);


            // Foreign key: ToStepId
            entity.HasOne(wt => wt.ToWorkflowStep)
                  .WithMany(ws => ws.ToWorkflowTransitions)
                  .HasForeignKey(wt => wt.ToStepId)
                  .OnDelete(DeleteBehavior.Restrict);

            // Foreign key: WorkflowId
            entity.HasOne(wt => wt.Workflow)
                  .WithMany(w => w.WorkflowTransitions)
                  .HasForeignKey(wt => wt.WorkflowId)
                  .OnDelete(DeleteBehavior.Cascade);
        });

        builder.Entity<DocumentMetaData>()
        .HasOne(d => d.DocumentMetaTag)
        .WithMany()
        .HasForeignKey(d => d.DocumentMetaTagId)
        .OnDelete(DeleteBehavior.NoAction);

        builder.Entity<DocumentSignature>(entity =>
        {
            entity.HasOne(ds => ds.Document)
                .WithMany()
                .HasForeignKey(ds => ds.DocumentId)
                .OnDelete(DeleteBehavior.Cascade);

            entity.HasOne(ds => ds.SignatureUser)
                .WithMany()
                .HasForeignKey(ds => ds.SignatureUserId)
                .OnDelete(DeleteBehavior.Restrict);
        });

        builder.Entity<EmployeeCourseSessionQuizAnswer>()
            .HasOne(e => e.QuizQuestion)
            .WithMany()
            .HasForeignKey(e => e.QuestionId)
            .OnDelete(DeleteBehavior.Restrict); // or NoAction

        builder.Entity<EmployeeCourseSessionQuizAnswerOption>()
            .HasOne(x => x.QuizQuestionOption)
            .WithMany()
            .HasForeignKey(x => x.OptionId)
            .OnDelete(DeleteBehavior.Restrict); // or .NoAction


        builder.Entity<Document>(entity =>
        {
            entity.Property(x => x.StorageType).HasDefaultValue(StorageType.LOCAL_STORAGE);
        });

        builder.Entity<DocumentUserPermission>().HasQueryFilter(p => !p.IsDeleted);
        builder.Entity<DocumentRolePermission>().HasQueryFilter(p => !p.IsDeleted);
        builder.Entity<User>().ToTable("Users");
        builder.Entity<Role>().ToTable("Roles");
        builder.Entity<RoleClaim>().ToTable("RoleClaims");
        builder.Entity<UserClaim>().ToTable("UserClaims");
        builder.Entity<UserLogin>().ToTable("UserLogins");
        builder.Entity<UserRole>().ToTable("UserRoles");
        builder.Entity<UserToken>().ToTable("UserTokens");
        builder.Entity<DocumentUserPermission>().ToTable("DocumentUserPermissions");
        builder.Entity<DocumentRolePermission>().ToTable("DocumentRolePermissions");
        builder.Entity<UserNotification>().ToTable("UserNotifications");

        builder.Entity<CategoryRolePermission>(entity =>
        {
            entity.HasOne(d => d.Category)
                .WithMany(p => p.CategoryRolePermissions)
                .HasForeignKey(d => d.CategoryId)
                .OnDelete(DeleteBehavior.ClientSetNull);

            entity.HasOne(d => d.Role)
                .WithMany(p => p.CategoryRolePermissions)
                .HasForeignKey(d => d.RoleId)
                .OnDelete(DeleteBehavior.ClientSetNull);
        });

        builder.Entity<CategoryUserPermission>(entity =>
        {
            entity.HasOne(d => d.Category)
                .WithMany(p => p.CategoryUserPermissions)
                .HasForeignKey(d => d.CategoryId)
                .OnDelete(DeleteBehavior.ClientSetNull);

            entity.HasOne(d => d.User)
                .WithMany(p => p.CategoryUserPermissions)
                .HasForeignKey(d => d.UserId)
                .OnDelete(DeleteBehavior.ClientSetNull);
        });

        builder.Entity<Audit>(entity =>
        {
            entity.HasOne(d => d.Auditor)
            .WithMany(p => p.Audits)
            .HasForeignKey(d => d.AuditorId)
            .OnDelete(DeleteBehavior.ClientSetNull);

            entity.HasMany(d => d.NonConformances)
            .WithOne(p => p.Audit)
            .HasForeignKey(d => d.AuditId)
            .OnDelete(DeleteBehavior.ClientSetNull);

            entity.HasMany(d => d.CapaRequests)
           .WithOne(p => p.Audit)
           .HasForeignKey(d => d.AuditId)
           .OnDelete(DeleteBehavior.ClientSetNull);

            entity.HasMany(d => d.AuditResponses)
               .WithOne(p => p.Audit)
               .HasForeignKey(d => d.AuditId)
               .OnDelete(DeleteBehavior.ClientSetNull);
            entity.HasMany(d => d.AuditReviewers)
                .WithOne(p => p.Audit)
                .HasForeignKey(d => d.AuditId)
                .OnDelete(DeleteBehavior.ClientSetNull);
        });

        builder.Entity<AuditResponse>(entity =>
        {
            entity.HasOne(d => d.Audit)
           .WithMany(p => p.AuditResponses)
           .HasForeignKey(d => d.AuditId)
           .OnDelete(DeleteBehavior.ClientSetNull);
        });

        builder.Entity<AuditReviewer>(entity =>
        {
            entity.HasOne(d => d.Reviewer)
           .WithMany()
           .HasForeignKey(d => d.ReviewerId)
           .OnDelete(DeleteBehavior.ClientSetNull);

            entity.HasOne(d => d.Submitter)
             .WithMany()
             .HasForeignKey(d => d.SubmitterId)
             .OnDelete(DeleteBehavior.ClientSetNull);
        });

        builder.Entity<AuditResponse>()
             .HasMany(ar => ar.CapaRequests)
             .WithOne(cr => cr.AuditResponse)
             .HasForeignKey(cr => cr.AuditResponseId)
             .OnDelete(DeleteBehavior.Restrict);

        builder.Entity<AuditResponse>()
            .HasMany(ar => ar.NonConformances)
            .WithOne(cr => cr.AuditResponse)
            .HasForeignKey(cr => cr.AuditResponseId)
            .OnDelete(DeleteBehavior.Restrict);

        builder.Entity<AuditTemplateQuestion>(entity =>
        {
            entity.HasOne(d => d.AuditTemplate)
                .WithMany(p => p.Questions)
                .HasForeignKey(d => d.AuditTemplateId)
                .OnDelete(DeleteBehavior.Cascade);

            entity.HasMany(d => d.QuestionOptions)
              .WithOne(p => p.AuditTemplateQuestion)
              .HasForeignKey(d => d.AuditTemplateQuestionId)
              .OnDelete(DeleteBehavior.Cascade);
        });

        builder.Entity<NonConformance>(entity =>
        {
            entity.HasOne(d => d.ReportedBy)
                .WithMany()
                .HasForeignKey(d => d.ReportedById)
                .OnDelete(DeleteBehavior.ClientSetNull);

            entity.HasMany(c => c.CapaRequests)
                .WithOne(e => e.NonConformance)
                .HasForeignKey(e => e.NonConformanceId)
                .OnDelete(DeleteBehavior.SetNull);
        });

        builder.Entity<NonConformanceResponse>(entity =>
        {
            entity.HasOne(d => d.NonConformance)
                .WithMany(p => p.NonConformanceResponses)
                .HasForeignKey(d => d.NonConformanceId)
                .OnDelete(DeleteBehavior.Cascade);

            entity.HasOne(d => d.ResponsiblePerson)
                .WithMany()
                .HasForeignKey(d => d.ResponsiblePersonId)
                .OnDelete(DeleteBehavior.ClientSetNull);

            entity.HasMany(d => d.NonConformanceRespAttachments)
             .WithOne(c => c.NonConformanceResponse)
             .HasForeignKey(d => d.NonConformanceResponseId)
             .OnDelete(DeleteBehavior.Cascade);

            entity.HasOne(d => d.VerifiedBy)
              .WithMany()
              .HasForeignKey(d => d.VerifiedById)
              .OnDelete(DeleteBehavior.ClientSetNull);
        });

        builder.Entity<CapaRequest>(entity =>
        {
            entity.HasOne(d => d.RaisedBY)
                .WithMany()
                .HasForeignKey(d => d.RaisedById)
                .OnDelete(DeleteBehavior.ClientSetNull);

            entity.HasOne(d => d.Audit)
              .WithMany(c => c.CapaRequests)
              .HasForeignKey(d => d.AuditId)
              .OnDelete(DeleteBehavior.ClientSetNull);

            entity.HasOne(d => d.Audit)
             .WithMany(c => c.CapaRequests)
             .HasForeignKey(d => d.AuditId)
             .OnDelete(DeleteBehavior.ClientSetNull);

        });
        builder.Entity<CapaRequest>(entity =>
        {
            entity.HasOne(d => d.RaisedBY)
                .WithMany()
                .HasForeignKey(d => d.RaisedById)
                .OnDelete(DeleteBehavior.ClientSetNull);

            entity.HasOne(d => d.Audit)
              .WithMany(c => c.CapaRequests)
              .HasForeignKey(d => d.AuditId)
              .OnDelete(DeleteBehavior.ClientSetNull);

            entity.HasMany(d => d.CapaActions)
             .WithOne(c => c.CapaRequest)
             .HasForeignKey(d => d.CapaRequestId)
             .OnDelete(DeleteBehavior.Cascade);

            entity.HasMany(d => d.CapaRootCauses)
            .WithOne(c => c.CapaRequest)
            .HasForeignKey(d => d.CapaRequestId)
            .OnDelete(DeleteBehavior.Cascade);

            entity.HasMany(d => d.CapaAttachments)
            .WithOne(c => c.CapaRequest)
            .HasForeignKey(d => d.CapaRequestId)
            .OnDelete(DeleteBehavior.Cascade);

            entity.HasMany(d => d.CapaReviews)
            .WithOne(c => c.CapaRequest)
            .HasForeignKey(d => d.CapaRequestId)
            .OnDelete(DeleteBehavior.Cascade);

        });

        builder.Entity<CapaAction>(entity =>
        {
            entity.HasOne(d => d.AssignTo)
                .WithMany()
                .HasForeignKey(d => d.AssignToId)
                .OnDelete(DeleteBehavior.ClientSetNull);
        });

        builder.Entity<CapaReview>(entity =>
        {
            entity.HasOne(d => d.ReviewedBy)
                .WithMany()
                .HasForeignKey(d => d.ReviewedById)
                .OnDelete(DeleteBehavior.ClientSetNull);
        });
        builder.Entity<CapaReview>(entity =>
        {
            entity.HasOne(d => d.ReviewedBy)
                .WithMany()
                .HasForeignKey(d => d.ReviewedById)
                .OnDelete(DeleteBehavior.ClientSetNull);
        });
        builder.Entity<CapaAttachmentChunk>(entity =>
        {
            entity.HasOne(d => d.CapaAttachment)
                .WithMany(d => d.CapaAttachmentChunks)
                .HasForeignKey(d => d.CapaAttachmentId)
                .OnDelete(DeleteBehavior.Cascade);
        });

        builder.Entity<NonConformanceRespAttachmentChunk>(entity =>
        {
            entity.HasOne(d => d.NonConformanceRespAttachment)
                .WithMany(d => d.NonConformanceRespAttachmentChunks)
                .HasForeignKey(d => d.NonConformanceRespAttachmentId)
                .OnDelete(DeleteBehavior.Cascade);
        });

        builder.Entity<Document>()
            .Property(e => e.IsArchive)
            .HasDefaultValue(false);


        builder.Entity<WorkflowTransitionRole>()
            .HasKey(o => new { o.WorkflowTransitionId, o.RoleId });

        builder.Entity<WorkflowTransitionUser>()
            .HasKey(o => new { o.WorkflowTransitionId, o.UserId });

        builder.Entity<EmployeeCourseSessionMedia>(entity =>
        {
            entity.HasOne(d => d.EmployeeCourseSession)
                .WithMany(p => p.EmployeeCourseSessionMedias)
                .HasForeignKey(d => d.EmployeeCourseSessionId)
                .OnDelete(DeleteBehavior.NoAction);
        });

        builder.Entity<EmployeeCourseSession>(entity =>
        {
            entity.HasOne(d => d.EmployeeCourse)
                .WithMany(p => p.EmployeeCourseSessions)
                .HasForeignKey(d => d.EmployeeCourseId)
                .OnDelete(DeleteBehavior.NoAction);
        });

        builder.Entity<RiskAttachment>(entity =>
        {
            entity.HasOne(d => d.UploadedBy)
                .WithMany()
                .HasForeignKey(d => d.UploadedById)
                .OnDelete(DeleteBehavior.NoAction);
        });

        builder.Entity<RiskControl>(entity =>
        {
            entity.HasOne(d => d.ResponsibleBy)
                .WithMany()
                .HasForeignKey(d => d.ResponsibleId)
                .OnDelete(DeleteBehavior.NoAction);
        });

        builder.Entity<RiskReview>(entity =>
        {
            entity.HasOne(d => d.Reviewer)
                .WithMany()
                .HasForeignKey(d => d.ReviewerId)
                .OnDelete(DeleteBehavior.NoAction);
        });

        builder.Entity<Risk>(entity =>
        {
            entity.HasOne(d => d.AssignedTo)
                .WithMany()
                .HasForeignKey(d => d.AssignedToId)
                .OnDelete(DeleteBehavior.NoAction);

            entity.HasOne(d => d.CreatedUser)
              .WithMany()
              .HasForeignKey(d => d.CreatedBy)
              .OnDelete(DeleteBehavior.NoAction);

            entity.HasOne(d => d.ModifiedUser)
            .WithMany()
            .HasForeignKey(d => d.ModifiedBy)
            .OnDelete(DeleteBehavior.NoAction);
        });

        builder.Entity<Complaint>(entity =>
        {
            entity.HasOne(d => d.AssignedTo)
                .WithMany()
                .HasForeignKey(d => d.AssignedToId)
                .OnDelete(DeleteBehavior.NoAction);

            entity.HasOne(d => d.CreatedUser)
              .WithMany()
              .HasForeignKey(d => d.CreatedBy)
              .OnDelete(DeleteBehavior.NoAction);

            entity.HasOne(d => d.ModifiedUser)
            .WithMany()
            .HasForeignKey(d => d.ModifiedBy)
            .OnDelete(DeleteBehavior.NoAction);
        });
        builder.Entity<ComplaintAction>(entity =>
        {
            entity.HasOne(d => d.Responsible)
                .WithMany()
                .HasForeignKey(d => d.ResponsibleId)
                .OnDelete(DeleteBehavior.NoAction);

        });
        builder.Entity<ComplaintAttachment>(entity =>
        {
            entity.HasOne(d => d.UploadedBy)
                .WithMany()
                .HasForeignKey(d => d.UploadedById)
                .OnDelete(DeleteBehavior.NoAction);

        });
        builder.Entity<ComplaintComment>(entity =>
        {
            entity.HasOne(d => d.CommentedBy)
                .WithMany()
                .HasForeignKey(d => d.CommentedById)
                .OnDelete(DeleteBehavior.NoAction);

        });
        builder.Entity<ComplaintInvestigation>(entity =>
        {
            entity.HasOne(d => d.Investigator)
                .WithMany()
                .HasForeignKey(d => d.InvestigatorId)
                .OnDelete(DeleteBehavior.NoAction);

        });
        builder.Entity<ComplaintRootCause>(entity =>
        {
            entity.HasOne(d => d.CreatedBy)
                .WithMany()
                .HasForeignKey(d => d.CreatedById)
                .OnDelete(DeleteBehavior.NoAction);

        });
        builder.Entity<SupplierAudit>(entity =>
        {
            entity.HasOne(d => d.Auditor)
                .WithMany()
                .HasForeignKey(d => d.AuditorId)
                .OnDelete(DeleteBehavior.NoAction);

        });
        builder.Entity<SupplierPerformance>(entity =>
        {
            entity.HasOne(d => d.EvaluatedBy)
                .WithMany()
                .HasForeignKey(d => d.EvaluatedById)
                .OnDelete(DeleteBehavior.NoAction);

        });
        builder.Entity<SupplierQualification>(entity =>
        {
            entity.HasOne(d => d.QualifiedBy)
                .WithMany()
                .HasForeignKey(d => d.QualifiedById)
                .OnDelete(DeleteBehavior.NoAction);

        });

        builder.Entity<AuditResponseLog>(entity =>
        {
            entity.HasOne(d => d.Audit)
                .WithMany()
                .HasForeignKey(d => d.AuditId)
                .OnDelete(DeleteBehavior.Restrict);

            entity.HasOne(d => d.CreatedByUser)
                .WithMany()
                .HasForeignKey(d => d.CreatedBy)
                .OnDelete(DeleteBehavior.NoAction);

        });

        builder.Entity<NonConformanceLog>(entity =>
        {

            entity.HasOne(d => d.CreatedByUser)
                .WithMany()
                .HasForeignKey(d => d.CreatedBy)
                .OnDelete(DeleteBehavior.NoAction);

            entity.HasOne(d => d.ReportedBy)
               .WithMany()
               .HasForeignKey(d => d.ReportedById)
               .OnDelete(DeleteBehavior.NoAction);

        });

        builder.Entity<RiskLog>(entity =>
        {
            entity.HasOne(d => d.CreatedByUser)
                .WithMany()
                .HasForeignKey(d => d.CreatedBy)
                .OnDelete(DeleteBehavior.NoAction);

            entity.HasOne(d => d.AssignedTo)
               .WithMany()
               .HasForeignKey(d => d.AssignedToId)
               .OnDelete(DeleteBehavior.NoAction);

        });

        builder.Entity<CapaRequestLog>(entity =>
        {
            entity.HasOne(d => d.CreatedByUser)
                .WithMany()
                .HasForeignKey(d => d.CreatedBy)
                .OnDelete(DeleteBehavior.NoAction);

            entity.HasOne(d => d.RaisedBY)
               .WithMany()
               .HasForeignKey(d => d.RaisedById)
               .OnDelete(DeleteBehavior.NoAction);

        });
        builder.Entity<ComplaintLog>(entity =>
        {
            entity.HasOne(d => d.AssignedTo)
                .WithMany()
                .HasForeignKey(d => d.AssignedToId)
                .OnDelete(DeleteBehavior.NoAction);

            entity.HasOne(d => d.CreatedByUser)
               .WithMany()
               .HasForeignKey(d => d.CreatedBy)
               .OnDelete(DeleteBehavior.NoAction);

        });



        builder.DefalutMappingValue();
        builder.DefalutDeleteValueFilter();
    }
}
