﻿using iText.Kernel.Colors;
using iText.Kernel.Pdf;
using iText.Layout.Properties;
using Microsoft.AspNetCore.Hosting;
using Microsoft.Extensions.Logging;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;


namespace QualityManagement.Helper
{
    public class EmployeeCertificateGenerator(
        IWebHostEnvironment webHostEnvironment,
        ILogger<EmployeeCertificateGenerator> logger)
    {
        public async Task<byte[]> GenerateCertificate(Guid id, string courseTitle, string employeeName, DateTime completedDate)
        {
            try
            {
                var filePath = Path.Combine(webHostEnvironment.WebRootPath, "Certificates", "Certificate.pdf");
                if (!File.Exists(filePath))
                {
                    logger.LogError("Certificate template file not found at path: {FilePath}", filePath);
                      return Array.Empty<byte>(); ;
                }
                var stream = new FileStream(filePath, FileMode.Open, FileAccess.Read);
                var bytes = AesOperation.ConvertStreamToByteArray(stream);
                using var pdfReaderStream = new MemoryStream(bytes);
                using var pdfWriterStream = new MemoryStream();
                var pdfReader = new PdfReader(pdfReaderStream);
                var pdfWriter = new PdfWriter(pdfWriterStream);
                var pdfDoc = new PdfDocument(pdfReader, pdfWriter);
                var document = new iText.Layout.Document(pdfDoc);
                var currentPageNumber = 1;

                var page = pdfDoc.GetPage(currentPageNumber);
                var pageWidth = page.GetPageSize().GetWidth();

                float paragraphWidth = 500;
                float x = (pageWidth - paragraphWidth) / 2;

                // Try several common Unicode fonts that support Romanian characters.
                var fontCandidates = new[]
                {
                    Path.Combine(webHostEnvironment.WebRootPath, "fonts", "DejaVuSans-Bold.ttf"),
                    Path.Combine(webHostEnvironment.WebRootPath, "fonts", "NotoSans-Bold.ttf"),
                    Path.Combine(webHostEnvironment.WebRootPath, "fonts", "LiberationSans-Bold.ttf"),
                    Path.Combine(webHostEnvironment.WebRootPath, "fonts", "ArialUnicodeMS-Bold.otf")
                };

                iText.Kernel.Font.PdfFont unicodeFont = null;
                foreach (var candidate in fontCandidates)
                {
                    try
                    {
                        if (File.Exists(candidate))
                        {
                            // Use IDENTITY_H encoding for full Unicode support and embed the font.
                            unicodeFont = iText.Kernel.Font.PdfFontFactory.CreateFont(candidate, iText.IO.Font.PdfEncodings.IDENTITY_H);
                            logger.LogDebug("Loaded font '{FontPath}' for PDF generation.", candidate);
                            break;
                        }
                    }
                    catch (Exception ex)
                    {
                        logger.LogWarning(ex, "Failed to load font candidate '{FontPath}' - trying next.", candidate);
                    }
                }

                // If no external font found, fall back to a standard font (won't render some diacritics).
                if (unicodeFont == null)
                {
                    logger.LogWarning("No Unicode TTF font found in wwwroot/fonts. Romanian diacritics may not render correctly. Please add a Unicode TTF (e.g., DejaVuSans.ttf) to wwwroot/fonts.");
                    unicodeFont = iText.Kernel.Font.PdfFontFactory.CreateFont(iText.IO.Font.Constants.StandardFonts.HELVETICA_BOLD);
                }
                var nameColor = new DeviceRgb(207, 176, 105); // #cfb069

                var employeeNameText = new iText.Layout.Element.Paragraph(employeeName)
                    .SetFont(unicodeFont)
                    .SetFontSize(30)
                    .SetFontColor(nameColor)
                    .SetTextAlignment(iText.Layout.Properties.TextAlignment.CENTER)
                    .SetFixedPosition(x, 310, paragraphWidth)
                    .SetPageNumber(currentPageNumber);
                document.Add(employeeNameText);

                var courseTitleText = new iText.Layout.Element.Paragraph(courseTitle)
                    .SetFont(unicodeFont)
                    .SetFontSize(22)
                    .SetFontColor(nameColor)
                    .SetTextAlignment(iText.Layout.Properties.TextAlignment.CENTER)
                    .SetFixedPosition(x, 195, paragraphWidth)
                    .SetPageNumber(currentPageNumber);

                document.Add(courseTitleText);

                var date = completedDate.Date.ToString("dd MMMM yyyy");
                var dateText = new iText.Layout.Element.Paragraph(date)
                    .SetFont(unicodeFont)
                    .SetFontSize(15)
                    .SetTextAlignment(iText.Layout.Properties.TextAlignment.CENTER)
                    .SetFixedPosition(x, 85, paragraphWidth)
                    .SetPageNumber(currentPageNumber);
                document.Add(dateText);
                document.Close();
                return pdfWriterStream.ToArray();
            }
            catch (Exception ex)
            {
                logger.LogError(ex, "Error occurred while adding signature to PDF.");
                return Array.Empty<byte>(); ;
            }
        }
    }
}
