﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;

public class AddAuditCommandHandler(
   IAuditRepository _auditRepository,
   IAuditTemplateRepository auditTemplateRepository,
   IAuditResponseLogRepository auditResponseLogRepository,
   IAuditReviewerRepository auditReviewerRepository,
   IMapper _mapper,
   IUnitOfWork<QMSDbContext> _uow,
   UserInfoToken userInfoToken,
   IUserRepository _userRepository,
   ISendEmailRepository sendEmailRepository,
   IUserNotificationRepository userNotificationRepository,
   ILogger<AddAuditCommandHandler> _logger) : IRequestHandler<AddAuditCommand, ServiceResponse<AuditDto>>
{
    public async Task<ServiceResponse<AuditDto>> Handle(AddAuditCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var existEntity = await _auditRepository.All
            .Where(c => c.AuditTemplateId == request.AuditTemplateId)
            .FirstOrDefaultAsync();
            if (existEntity != null)
            {
                if (existEntity.Status == AUDIT_STATUS.CREATED
                    || existEntity.Status == AUDIT_STATUS.INPROGRESS
                    || existEntity.Status == AUDIT_STATUS.REWORK)
                {
                    return ServiceResponse<AuditDto>.Return409("Audit status is already 'Created' or 'In Progress'");
                }
            }
            var entity = _mapper.Map<Audit>(request);
            entity.Id = Guid.NewGuid();
            entity.AuditNumber = await _auditRepository.GenerateAuditNumberAsync();
            var totalMaxScore = await auditTemplateRepository.AllIncluding(c => c.Questions)
               .Where(c => c.Id == request.AuditTemplateId)
               .SelectMany(c => c.Questions)
               .SumAsync(q => q.MaxScore);
            if (totalMaxScore > 0)
            {
                entity.TotalScore = totalMaxScore;
            }
            _auditRepository.Add(entity);
            var auditResponseLog = _mapper.Map<AuditResponseLog>(entity);
            auditResponseLog.Id = Guid.NewGuid();
            auditResponseLog.CreatedBy = userInfoToken.Id;
            auditResponseLog.CreatedDate = DateTime.UtcNow;
            auditResponseLog.LogStatus = LogStatus.Created;
            auditResponseLogRepository.Add(auditResponseLog);
            var notification = new NotificationAuditDto
            {
                AuditId = entity.Id,
                UserId = userInfoToken.Id,
                Message = $"You have been assigned the audit \"{entity.Title}.\"",
                Title = entity.Title,
                NotificationsType =  NotificationsType.AUDIT
            };
            userNotificationRepository.AddUserNotificationByAudit(notification);
            var auditorUserInfo = await _userRepository.FindAsync(request.AuditorId);
            var currentUserInfo = _userRepository.Find(userInfoToken.Id);
            sendEmailRepository.AddAuditEmails(new SendEmail
            {
                Email = auditorUserInfo.Email,
                FromEmail = currentUserInfo.Email,
                FromName = currentUserInfo.FirstName + ' ' + currentUserInfo.LastName,
                ToName = auditorUserInfo.FirstName + ' ' + auditorUserInfo.LastName,
                CreatedBy = userInfoToken.Id,
                CreatedDate = DateTime.UtcNow,
            }, $"You have been assigned the audit \"{entity.Title}.\"");

            var auditReview = new AuditReviewer
            {
                Id = Guid.NewGuid(),
                AuditId = entity.Id,
                Comments = "",
                AuditStatus = entity.Status,
                ReviewedOn = DateTime.UtcNow,
                SubmitterId = userInfoToken.Id
            };

            auditReviewerRepository.Add(auditReview);

            if (await _uow.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<AuditDto>.Return500();
            }
            var entityDto = _mapper.Map<AuditDto>(entity);
            return ServiceResponse<AuditDto>.ReturnResultWith201(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Erorr while saving Audit");
            return ServiceResponse<AuditDto>.Return500("Error while Saving Audit");
        }
    }
}
