﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;
public class GetAuditReportDetailsCommandHandler(
    IAuditRepository _auditRepository,
    IAuditTemplateQuestionRepository auditTemplateQuestionRepository,
    IMapper _mapper,
    ILogger<GetAuditCommandHandler> _logger) : IRequestHandler<GetAuditReportDetailsCommand, ServiceResponse<AuditReportDetailsDto>>
{
    public async Task<ServiceResponse<AuditReportDetailsDto>> Handle(GetAuditReportDetailsCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var existEntity = await _auditRepository.All
               .Include(c => c.AuditTemplate)
                   .ThenInclude(a => a.Questions)
                   .ThenInclude(c => c.QuestionOptions)
               .Include(c => c.AuditTemplate)
                   .ThenInclude(c => c.Department)
               .Include(c => c.Auditor)
               .Include(c => c.NonConformances)
                   .ThenInclude(c => c.NonConformanceResponses)
                    .ThenInclude(c => c.ResponseType)
              .Include(c => c.NonConformances)
                  .ThenInclude(c => c.NonConformanceResponses)
                      .ThenInclude(c => c.VerifiedBy)
              .Include(c => c.NonConformances)
                  .ThenInclude(c => c.NonConformanceResponses)
                    .ThenInclude(c => c.ResponsiblePerson)
               .Include(c => c.NonConformances)
                   .ThenInclude(c => c.ReportedBy)
               .Include(c => c.CapaRequests)
                    .ThenInclude(c => c.RaisedBY)
               .Include(c => c.CapaRequests)
                   .ThenInclude(ca => ca.CapaActions)
                        .ThenInclude(ca => ca.AssignTo)
               .Include(c => c.CapaRequests)
                   .ThenInclude(crc => crc.CapaRootCauses)
                        .ThenInclude(crc => crc.CreatedBy)
               .Include(c => c.CapaRequests)
                    .ThenInclude(crc => crc.CapaRootCauses)
                        .ThenInclude(crcm => crcm.CapaRootCauseMethodology)
               .Include(c => c.CapaRequests)
                    .ThenInclude(ca => ca.CapaAttachments)
               .Include(c => c.CapaRequests)
                    .ThenInclude(c => c.CapaReviews)
                .Include(c => c.CapaRequests)
                    .ThenInclude(c => c.CapaReviews)
                        .ThenInclude(c => c.ReviewedBy)
               .Include(c => c.AuditResponses)
                 .ThenInclude(c => c.AuditResponseAttachments)
               .Include(c => c.AuditReviewers.OrderByDescending(c => c.ReviewedOn))
                    .ThenInclude(c => c.Reviewer)
              .Include(c => c.AuditReviewers.OrderByDescending(c => c.ReviewedOn))
                    .ThenInclude(c => c.Submitter)
               .Where(a => a.Id == request.Id)
                .AsNoTracking()
                .AsSplitQuery()
               .FirstOrDefaultAsync();

            if (existEntity == null)
            {
                return ServiceResponse<AuditReportDetailsDto>.Return404();
            }

            var entityDto = _mapper.Map<AuditReportDetailsDto>(existEntity);

            entityDto.AuditQuestionResponses = existEntity.AuditResponses.Select(auditResponse =>
            {
                var question = auditTemplateQuestionRepository.AllIncluding(c => c.QuestionOptions)
                    .Where(x => x.Id == auditResponse.QuestionId)
                    .FirstOrDefault();

                if (question != null &&
                    (question.InputType == AUDIT_TEMPLATE_QUESTION_TYPE.SINGLE_SELECT ||
                     question.InputType == AUDIT_TEMPLATE_QUESTION_TYPE.MULTI_SELECT))
                {
                    var responseIds = auditResponse.Response.Split(',')
                        .Select(id => Guid.Parse(id.Trim()))
                        .ToList();

                    var selectedOptions = question.QuestionOptions
                        .Where(c => responseIds.Contains(c.Id))
                        .Select(c => c.OptionText)
                        .ToList();

                    return new AuditQuestionResponseDto
                    {
                        QuestionId = question.Id,
                        Question = question.Question,
                        InputType = question.InputType,
                        EffectiveScore = auditResponse.Score,
                        TotalScore = question.MaxScore,
                        Response = string.Join(", ", selectedOptions),
                        AuditResponseAttachments = auditResponse.AuditResponseAttachments?.Where(c => !c.IsDeleted)?.Select(a => new AuditResponseAttachmentDto
                        {
                            Id = a.Id,
                            FileName = a.FileName,
                            Extension = a.Extension,
                            TotalChunk = a.TotalChunk,
                            FileSize = a.FileSize
                        }).ToList()
                    };
                }

                return new AuditQuestionResponseDto
                {
                    QuestionId = auditResponse.QuestionId,
                    Question = question.Question,
                    InputType = question.InputType,
                    EffectiveScore = auditResponse.Score,
                    TotalScore = question.MaxScore,
                    Response = auditResponse.Response,
                    AuditResponseAttachments = auditResponse.AuditResponseAttachments?.Where(c => !c.IsDeleted)?.Select(a => new AuditResponseAttachmentDto
                    {
                        Id = a.Id,
                        FileName = a.FileName,
                        Extension = a.Extension,
                        TotalChunk = a.TotalChunk,
                        FileSize = a.FileSize
                    }).ToList()
                };
            }).ToList();

            return ServiceResponse<AuditReportDetailsDto>.ReturnResultWith200(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Error while getting Audit");
            return ServiceResponse<AuditReportDetailsDto>.Return500("Error while getting Audit");
        }
    }
}

