﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;
public class UpdateAuditCommandHandler(
    IAuditRepository auditRepository,
    IUnitOfWork<QMSDbContext> _uow,
    UserInfoToken userInfoToken,
    IAuditResponseLogRepository auditResponseLogRepository,
    IAuditReviewerRepository auditReviewerRepository,
    IMapper _mapper,
    ILogger<UpdateActionCommandHandler> _logger
    ) : IRequestHandler<UpdateAuditCommand, ServiceResponse<AuditDto>>
{
    public async Task<ServiceResponse<AuditDto>> Handle(UpdateAuditCommand request, CancellationToken cancellationToken)
    {
        var entityExist = await auditRepository.FindBy(v => v.AuditTemplateId == request.AuditTemplateId && v.Id == request.Id).FirstOrDefaultAsync();

        if (entityExist == null)
        {
            _logger.LogError("Audit Not Found.");
            return ServiceResponse<AuditDto>.Return404("Audit Not Found.");
        }
        if (request.Status != entityExist.Status)
        {
            var auditReview = new AuditReviewer
            {
                Id = Guid.NewGuid(),
                AuditId = entityExist.Id,
                Comments = "",
                AuditStatus = entityExist.Status,
                ReviewedOn = DateTime.UtcNow,
                ReviewerId = userInfoToken.Id
            };
            auditReviewerRepository.Add(auditReview);
        }
        entityExist.Status = request.Status;
        auditRepository.Update(entityExist);
        var auditResponseLog = _mapper.Map<AuditResponseLog>(entityExist);
        auditResponseLog.Id = Guid.NewGuid();
        auditResponseLog.CreatedBy = userInfoToken.Id;
        auditResponseLog.CreatedDate = DateTime.UtcNow;
        auditResponseLog.LogStatus = LogStatus.Updated;
        auditResponseLogRepository.Add(auditResponseLog);


        if (await _uow.SaveAsync() <= 0)
        {
            return ServiceResponse<AuditDto>.Return500();
        }
        var entityDto = new AuditDto
        {
            Id = entityExist.Id,
            Status = entityExist.Status,
        };
        //entityDto.AuditStatus = entityExist.Status;
        return ServiceResponse<AuditDto>.ReturnResultWith200(entityDto);
    }
}
