﻿using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Helper;
using QualityManagement.Repository;
using QualityManagement.Repository.RiskManagment;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class DownloadAuditResponseAttachmentChunkCommandHandler(
        IDocumentTokenRepository _documentTokenRepository,
        IAuditResponseAttachmentRepository _auditResponseAttachmentRepository,
        IAuditResponseAttachmentChunkRepository _auditResponseAttachmentChunkRepository,
        ILogger<DownloadRiskAttachmentChunkCommandHandler> _logger,
        IMediator _mediator) : IRequestHandler<DownloadAuditResponseAttachmentChunkCommand, ServiceResponse<DocumentDownload>>
    {
        public async Task<ServiceResponse<DocumentDownload>> Handle(DownloadAuditResponseAttachmentChunkCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var isTokenFound = await _documentTokenRepository.All.AnyAsync(c => c.DocumentId == request.Id && c.Token == request.Token);
                if (!isTokenFound)
                {
                    return ServiceResponse<DocumentDownload>.Return404("Token not found");
                }
                var auditAttachment = await _auditResponseAttachmentRepository.All.Where(c => c.Id == request.Id).FirstOrDefaultAsync();
                if (auditAttachment == null)
                {
                    return ServiceResponse<DocumentDownload>.Return404("Audit Response attachment not found");
                }
                var auditAttachmentChunks = await _auditResponseAttachmentChunkRepository.All.Where(c => c.AuditResponseAttachmentId == auditAttachment.Id).OrderBy(c => c.ChunkIndex).ToListAsync();
                if (auditAttachmentChunks.Count <= 0)
                {
                    return ServiceResponse<DocumentDownload>.Return404("Audit Response attachment chunk not found");
                }
                var chunks = auditAttachmentChunks.Select(c => new FileChunk
                {
                    ChunkIndex = c.ChunkIndex,
                    Url = c.Url,
                }).ToList();
                var fileDownloadRequestCommand = new FileDownloadRequestCommand
                {
                    StorageSettingId = auditAttachment.StorageSettingId,
                    Key = auditAttachment.Key,
                    IV = auditAttachment.IV,
                    Extension = auditAttachment.Extension,
                    Chunks = chunks,
                    FileName = auditAttachment.FileName,
                };
                var result = await _mediator.Send(fileDownloadRequestCommand);
                return result;
            }
            catch (Exception ex)
            {
                _logger.LogError(ex, "error while getting Audit Response attachmentt");
                return ServiceResponse<DocumentDownload>.Return500("An unexpected fault happened. Try again later. ");
            }
        }
    }
}
