﻿using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;
public class MarkAuditResponseAttachmentChunksUploadedCommandHandler(
    IAuditResponseAttachmentRepository auditResponseAttachmentRepository,
    IAuditResponseAttachmentChunkRepository auditResponseAttachmentChunkRepository,
    IUnitOfWork<QMSDbContext> uow,
    IStorageSettingRepository _storageSettingRepository,
    StorageServiceFactory _storageServiceFactory,
    ILogger<MarkAttachmentChunksUploadedCommandHandler> _logger) : IRequestHandler<MarkAuditResponseAttachmentChunksUploadedCommand, ServiceResponse<AuditResponseAttachmentChunkStatus>>
{
    public async Task<ServiceResponse<AuditResponseAttachmentChunkStatus>> Handle(MarkAuditResponseAttachmentChunksUploadedCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var auditResponseAttachment = await auditResponseAttachmentRepository.All
                .Where(c => c.Id == request.AuditResponseAttachmentChunkId).FirstOrDefaultAsync();

            if (auditResponseAttachment == null)
            {
                return ServiceResponse<AuditResponseAttachmentChunkStatus>.ReturnFailed(404, "AuditResponseAttachment not found.");
            }

            var auditResponseChunks = auditResponseAttachmentChunkRepository
                .All
                .Where(c => c.AuditResponseAttachmentId == auditResponseAttachment.Id)
                .OrderByDescending(c => c.ChunkIndex)
                .ToList();

            if (auditResponseChunks.Count == 0)
            {
                return ServiceResponse<AuditResponseAttachmentChunkStatus>.ReturnFailed(404, " AuditResponseAttachment Chunks not found.");
            }


            if (auditResponseAttachment.TotalChunk == auditResponseChunks.Count())
            {
                auditResponseAttachment.IsAllChunkUploaded = request.Status;

                auditResponseAttachmentRepository.Update(auditResponseAttachment);

                if (await uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<AuditResponseAttachmentChunkStatus>.ReturnFailed(500, "Error While Adding NCAttachment");
                }
                var response = new AuditResponseAttachmentChunkStatus { AuditResponseAttachmentId = auditResponseAttachment.Id, Status = request.Status };
                return ServiceResponse<AuditResponseAttachmentChunkStatus>.ReturnResultWith200(response);
            }
            else
            {
                if (auditResponseChunks.Count() > 0)
                {
                    foreach (var documentChunk in auditResponseChunks)
                    {
                        var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(auditResponseAttachment.StorageSettingId);
                        var storageService = _storageServiceFactory.GetStorageService(storeageSetting.StorageType);
                        await storageService.DeleteFileAsync(documentChunk.Url, storeageSetting.JsonValue);

                    }
                    auditResponseAttachmentChunkRepository.RemoveRange(auditResponseChunks);
                }

                auditResponseAttachmentRepository.Remove(auditResponseAttachment);
                if (await uow.SaveAsync() <= 0)
                {
                    return ServiceResponse<AuditResponseAttachmentChunkStatus>.ReturnFailed(500, "Error While Adding AuditResponseAttachment");
                }

                return ServiceResponse<AuditResponseAttachmentChunkStatus>.ReturnFailed(500, "Upload failed: Some parts of the document were not uploaded successfully. The document has been deleted. Please try uploading it again.");
            }
        }
        catch (Exception ex)
        {
            _logger.LogError(ex, "Error while marking AuditResponseAttachment chunks uploaded.");
            return ServiceResponse<AuditResponseAttachmentChunkStatus>.ReturnFailed(500, "Error while marking AuditResponseAttachment chunks uploaded.");
        }
    }
}