﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;
public class AddAuditReviewCommandHandler(
    IAuditReviewerRepository auditReviewerRepository,
   IMapper _mapper,
   IAuditRepository auditRepository,
   UserInfoToken userInfoToken,
   IAuditResponseLogRepository auditResponseLogRepository,
   IUserNotificationRepository userNotificationRepository,
   IUserRepository userRepository,
   ISendEmailRepository sendEmailRepository,
   IUnitOfWork<QMSDbContext> _uow,
   ILogger<AddAuditReviewCommandHandler> _logger) : IRequestHandler<AddAuditReviewCommand, ServiceResponse<AuditReviewerDto>>
{
    public async Task<ServiceResponse<AuditReviewerDto>> Handle(AddAuditReviewCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var audit = await auditRepository.FindAsync(request.AuditId);
            if (audit == null)
            {
                return ServiceResponse<AuditReviewerDto>.Return404("Audit not found");
            }

            audit.Status = request.Status;
            auditRepository.Update(audit);
            var auditResponseLog = _mapper.Map<AuditResponseLog>(audit);
            auditResponseLog.Id = Guid.NewGuid();
            auditResponseLog.CreatedBy = userInfoToken.Id;
            auditResponseLog.CreatedDate = DateTime.UtcNow;
            auditResponseLog.LogStatus = LogStatus.Updated;
            auditResponseLogRepository.Add(auditResponseLog);

            var lastAuditReview = await auditReviewerRepository.All.OrderByDescending(c => c.ReviewedOn)
                .Where(c => c.AuditId == audit.Id)
                .FirstOrDefaultAsync();

            var auditReview = new AuditReviewer
            {
                Id = Guid.NewGuid(),
                AuditId = request.AuditId,
                Comments = request.Comments,
                AuditStatus = request.Status,
                ReviewedOn = DateTime.UtcNow,
                ReviewerId = lastAuditReview?.ReviewerId ?? userInfoToken.Id, // Use the last reviewer's ID or current user if no previous review exists
                SubmitterId = userInfoToken.Id
            };
            auditReviewerRepository.Add(auditReview);
            if (request.Status == AUDIT_STATUS.REJECTED || request.Status == AUDIT_STATUS.REWORK)
            {
                var notificationMessage = request.Status == AUDIT_STATUS.REJECTED
                        ? $"The audit \"{audit.Title}\" has been rejected. Reason: {auditReview.Comments}"
                        : request.Status == AUDIT_STATUS.REWORK
                            ? $"The audit \"{audit.Title}\" has been sent back for rework. Reason: {auditReview.Comments}"
                            : null;

                var notification = new NotificationAuditDto
                {
                    AuditId = audit.Id,
                    UserId = userInfoToken.Id,
                    Message = notificationMessage,
                    Title = audit.Title,
                    NotificationsType = request.Status == AUDIT_STATUS.REJECTED ? NotificationsType.AUDITREJECTED : NotificationsType.AUDITREWORK,
                };
                userNotificationRepository.AddUserNotificationByAudit(notification);

                var auditorUserInfo = await userRepository.FindAsync(audit.AuditorId);
                var currentUserInfo = await userRepository.FindAsync(userInfoToken.Id);

                sendEmailRepository.AddAuditEmails(new SendEmail
                {
                    Email = auditorUserInfo.Email,
                    FromEmail = currentUserInfo.Email,
                    FromName = currentUserInfo.FirstName + ' ' + currentUserInfo.LastName,
                    ToName = auditorUserInfo.FirstName + ' ' + auditorUserInfo.LastName,
                    CreatedBy = userInfoToken.Id,
                    CreatedDate = DateTime.UtcNow,
                }, notificationMessage);
            }
            if (await _uow.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<AuditReviewerDto>.Return500();
            }
            var entityDto = _mapper.Map<AuditReviewerDto>(auditReview);
            return ServiceResponse<AuditReviewerDto>.ReturnResultWith201(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Erorr while saving AuditReview");
            return ServiceResponse<AuditReviewerDto>.Return500("Error while Saving AuditReview");
        }
    }
}
