﻿using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;

public class UpdateAuditTemplateCommandHandler(
    IAuditTemplateRepository _auditTemplateRepository,
    IAuditRepository auditRepository,
    IAuditTemplateQuestionRepository auditTemplateQuestionRepository,
    IUnitOfWork<QMSDbContext> _uow,
    ILogger<UpdateAuditTemplateCommandHandler> _logger,
    IMapper _mapper,
    IAuditResponseRepository _auditResponseRepository) : IRequestHandler<UpdateAuditTemplateCommand, ServiceResponse<AuditTemplateDto>>
{
    public async Task<ServiceResponse<AuditTemplateDto>> Handle(UpdateAuditTemplateCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var existEntity = await _auditTemplateRepository
                .All.Where(c => c.Id == request.Id)
                .Include(c => c.Questions)
                .ThenInclude(c => c.QuestionOptions)
                .AsNoTracking()
                .FirstOrDefaultAsync();

            if (existEntity == null)
            {
                return ServiceResponse<AuditTemplateDto>.Return404();
            }
            var audit = await auditRepository.All.Where(c => c.AuditTemplateId == existEntity.Id && (c.Status == Data.Entities.AUDIT_STATUS.CREATED || c.Status == Data.Entities.AUDIT_STATUS.INPROGRESS)).FirstOrDefaultAsync();

            // Check Audit have Response
            if (audit != null)
            {
                var auditResponse = await _auditResponseRepository.All.Where(c => c.AuditId == audit.Id).FirstOrDefaultAsync();
                if (auditResponse != null)
                {
                    return ServiceResponse<AuditTemplateDto>.Return409($"Audit has already started on {audit.Title} and contains responses. Please create a new template to proceed.");
                }
            }
            if (existEntity.Questions.Count > 0)
            {
                auditTemplateQuestionRepository.RemoveRange(existEntity.Questions);
            }
            _mapper.Map(request, existEntity);
            _auditTemplateRepository.Update(existEntity);
            if (audit != null)
            {
                audit.TotalScore = existEntity.Questions.Sum(q => q.MaxScore);
                auditRepository.Update(audit);
            }
            auditTemplateQuestionRepository.AddRange(existEntity.Questions);
            if (await _uow.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<AuditTemplateDto>.Return500();
            }
            var entityDto = _mapper.Map<AuditTemplateDto>(existEntity);
            return ServiceResponse<AuditTemplateDto>.ReturnResultWith201(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Error while updating AuditTemplate");
            return ServiceResponse<AuditTemplateDto>.Return500("Error while updating AuditTemplate");
        }
    }
}
