﻿using Amazon.Runtime.Internal.Util;
using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class UpdateCapaActionCommandHandler(
        ICapaActionRepository _capaActionRepository,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<UpdateCapaActionCommandHandler> _logger,
        IUserRepository _userRepository,
        UserInfoToken _userInfoToken,
         IEmailSMTPSettingRepository _emailSMTPSettingRepository,
    ISendEmailRepository _sendEmailRepository) : IRequestHandler<UpdateCapaActionCommand, ServiceResponse<CapaActionDto>>
    {
        public async Task<ServiceResponse<CapaActionDto>> Handle(UpdateCapaActionCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var entity = await _capaActionRepository.All.Where(c => c.Id == request.Id).FirstOrDefaultAsync();
                if (entity == null)
                {
                    return ServiceResponse<CapaActionDto>.Return404("capa action not found");
                }
                _mapper.Map(request, entity);
                if (request.Status == CapaActionStatus.COMPLETED)
                {
                    entity.CompletedDate = DateTime.UtcNow;
                }
                _capaActionRepository.Update(entity);
                var defaultSmtp = await _emailSMTPSettingRepository.FindBy(c => c.IsDefault).FirstOrDefaultAsync();

                var userIds = new[] { _userInfoToken.Id, entity.AssignToId };
                var users = await _userRepository.All
                    .Where(u => userIds.Contains(u.Id))
                    .ToListAsync();
                var currentUser = users.FirstOrDefault(u => u.Id == _userInfoToken.Id);
                var assignedUser = users.FirstOrDefault(u => u.Id == request.AssignToId);
                try
                {
                    if (defaultSmtp != null && assignedUser != null && entity.AssignToId != request.AssignToId)
                    {
                        var emailmessage = "You have been assigned a new Corrective and Preventive Action (CAPA).";
                        var subject = "Corrective and Preventive Action (CAPA)Assignment";
                        var mainTitle = "CAPA Action Assignment Notification";
                        _sendEmailRepository.AddAsignmentEmails(new SendEmail
                        {
                            Email = assignedUser.Email,
                            FromEmail = defaultSmtp.FromEmail,
                            FromName = currentUser.FirstName + ' ' + currentUser.LastName,
                            ToName = assignedUser.FirstName + ' ' + assignedUser.LastName,
                            CreatedBy = currentUser.Id,
                            CreatedDate = DateTime.UtcNow,
                        }, emailmessage, mainTitle, subject);
                    }
                }
                catch (System.Exception ex)
                {
                    _logger.LogError(ex, "email does not sent");
                }
                if (await _uow.SaveAsync() <= 0)
                {
                    return ServiceResponse<CapaActionDto>.Return500("error while updating capa Action");
                }
                if (currentUser != null)
                {
                    entity.CreatedByUser = currentUser;
                }
                if (assignedUser != null)
                {
                    entity.AssignTo = assignedUser;
                }
                var dto = _mapper.Map<CapaActionDto>(entity);
                return ServiceResponse<CapaActionDto>.ReturnResultWith200(dto);
            }
            catch (Exception ex)
            {
                _logger.LogError(ex, "error while updating capa Action");
                return ServiceResponse<CapaActionDto>.Return500("error while updating capa Action");
            }
        }
    }
}
