﻿using MediatR;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Data.Dto;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class DownloadCapaAttachmentCommandHandler(
        ICapaAttachmentRepository _capaAttachmentRepository,
         ICapaAttachmentChunkRepository _capaAttachmentChunkRepository,
        IStorageSettingRepository _storageSettingRepository,
        StorageServiceFactory _storageServiceFactory) : IRequestHandler<DownloadCapaAttachmentCommand, ServiceResponse<CapaAttachmentDownload>>
    {

        public async Task<ServiceResponse<CapaAttachmentDownload>> Handle(DownloadCapaAttachmentCommand request, CancellationToken cancellationToken)
        {
            var capaAttachment = _capaAttachmentRepository.Find(request.CapaAttachmentId);
            if (capaAttachment == null)
            {
                return ServiceResponse<CapaAttachmentDownload>.ReturnFailed(404, "CapaAttachment is not found");
            }

            var capaAttachmentChunk = await _capaAttachmentChunkRepository.All
                .Where(c => c.CapaAttachmentId == request.CapaAttachmentId && c.ChunkIndex == request.ChunkIndex)
                .FirstOrDefaultAsync();

            if (capaAttachmentChunk == null)
            {
                return ServiceResponse<CapaAttachmentDownload>.ReturnFailed(404, "CapaAttachment Chunk is not found");
            }

            var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(capaAttachment.StorageSettingId);

            if (storeageSetting == null)
            {
                return ServiceResponse<CapaAttachmentDownload>.ReturnFailed(404, "Storage setting not found");
            }
            var storageService = _storageServiceFactory.GetStorageService(storeageSetting.StorageType);
            var fileResult = await storageService.DownloadFileAsync(capaAttachmentChunk.Url, storeageSetting.JsonValue, capaAttachment.Key, capaAttachment.IV);
            if (string.IsNullOrWhiteSpace(fileResult.ErrorMessage))
            {
                var extension = capaAttachmentChunk.Extension;
                if (!extension.StartsWith("."))
                    extension = "." + extension;
                CapaAttachmentDownload capaAttachmentDownload = new CapaAttachmentDownload
                {
                    Data = Convert.ToBase64String(fileResult.FileBytes),
                    ContentType = FileHelper.GetMimeTypeByExtension(extension),
                    FileName = capaAttachmentChunk.Url,
                    ChunkIndex = capaAttachmentChunk.ChunkIndex
                };
                return ServiceResponse<CapaAttachmentDownload>.ReturnResultWith200(capaAttachmentDownload);
            }
            return ServiceResponse<CapaAttachmentDownload>.ReturnFailed(500, fileResult.ErrorMessage);
        }
    }
}
