﻿using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Helper;
using QualityManagement.Repository;
using QualityManagement.Repository.RiskManagment;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class DownloadCapaAttachmentChunkCommandHandler(
        IDocumentTokenRepository _documentTokenRepository,
        ICapaAttachmentRepository capaAttachmentRepository,
        ICapaAttachmentChunkRepository capaAttachmentChunkRepository,
        ILogger<DownloadRiskAttachmentChunkCommandHandler> _logger,
        IMediator _mediator) : IRequestHandler<DownloadCapaAttachmentChunkCommand, ServiceResponse<DocumentDownload>>
    {
        public async Task<ServiceResponse<DocumentDownload>> Handle(DownloadCapaAttachmentChunkCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var isTokenFound = await _documentTokenRepository.All.AnyAsync(c => c.DocumentId == request.Id && c.Token == request.Token);
                if (!isTokenFound)
                {
                    return ServiceResponse<DocumentDownload>.Return404("Token not found");
                }
                var capaAttachment = await capaAttachmentRepository.All.Where(c => c.Id == request.Id).FirstOrDefaultAsync();
                if (capaAttachment == null)
                {
                    return ServiceResponse<DocumentDownload>.Return404("capa  attachment not found");
                }
                var capaAttachmentChunk = await capaAttachmentChunkRepository.All.Where(c => c.CapaAttachmentId == capaAttachment.Id).OrderBy(c => c.ChunkIndex).ToListAsync();
                if (capaAttachmentChunk.Count <= 0)
                {
                    return ServiceResponse<DocumentDownload>.Return404("capa attachment chunk not found");
                }
                var chunks = capaAttachmentChunk.Select(c => new FileChunk
                {
                    ChunkIndex = c.ChunkIndex,
                    Url = c.Url,
                }).ToList();
                var fileDownloadRequestCommand = new FileDownloadRequestCommand
                {
                    StorageSettingId = capaAttachment.StorageSettingId,
                    Key = capaAttachment.Key,
                    IV = capaAttachment.IV,
                    Extension = capaAttachment.Extension,
                    Chunks = chunks,
                    FileName = capaAttachment.FileName,
                };
                var result = await _mediator.Send(fileDownloadRequestCommand);
                return result;
            }
            catch (Exception ex)
            {
                _logger.LogError(ex, "error while getting capa attachment");
                return ServiceResponse<DocumentDownload>.Return500("An unexpected fault happened. Try again later. ");
            }
        }
    }
}
