﻿using AutoMapper;
using MediatR;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class UploadCapaAttechmentChunkCommandHandler(
        ICapaAttachmentRepository _capaAttachmentRepository,
        ICapaAttachmentChunkRepository _capaAttachmentChunkRepository,
        IStorageSettingRepository _storageSettingRepository,
        StorageServiceFactory _storageServiceFactory,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<UploadCapaAttechmentChunkCommandHandler> _logger) : IRequestHandler<UploadCapaAttechmentChunkCommand, ServiceResponse<CapaAttachmentChunkDto>>
    {
        public async Task<ServiceResponse<CapaAttachmentChunkDto>> Handle(UploadCapaAttechmentChunkCommand request, CancellationToken cancellationToken)
        {
            try
            {
                if (request.File == null)
                {
                    return ServiceResponse<CapaAttachmentChunkDto>.ReturnFailed(409, "Please select the file.");
                }

                var existCapaAttachment = _capaAttachmentRepository.Find(request.CapaAttachmentId);
                if (existCapaAttachment == null)
                {
                    return ServiceResponse<CapaAttachmentChunkDto>.Return404("CapaAttachment not found");
                }
                var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(existCapaAttachment.StorageSettingId);

                var storageService = _storageServiceFactory.GetStorageService(storeageSetting.StorageType);

                var fileNameKeyValut = await storageService.UploadFileChunkAsync(request.File, storeageSetting, existCapaAttachment.Extension, existCapaAttachment.Key, existCapaAttachment.IV);

                if (string.IsNullOrEmpty(fileNameKeyValut.FileName))
                {
                    return ServiceResponse<CapaAttachmentChunkDto>.Return422("Settings are not properly setup.");
                }
                var entity = new CapaAttachmentChunk()
                {
                    Id = Guid.NewGuid(),
                    ChunkIndex = request.ChunkIndex,
                    CapaAttachmentId = request.CapaAttachmentId,
                    Extension = request.Extension,
                    Size = request.Size,
                    Url = fileNameKeyValut.FileName
                };
                _capaAttachmentChunkRepository.Add(entity);
                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<CapaAttachmentChunkDto>.ReturnFailed(500, "Error While Added Document");
                }

                var entityDto = _mapper.Map<CapaAttachmentChunkDto>(entity);
                return ServiceResponse<CapaAttachmentChunkDto>.ReturnResultWith200(entityDto);

            }
            catch (System.Exception ex)

            {
                _logger.LogError(ex, "Error while storing CapaAttachmentChunk");
                return ServiceResponse<CapaAttachmentChunkDto>.Return500("Error while storing CapaAttachmentChunk.");
            }
        }
    }
}
