﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class UpdateCapaRequestCommandHandler(
        ICapaRequestRepository _capaRequestRepository,
        ICapaRequestLogRepository _capaRequestLogRepository,
        UserInfoToken userInfoToken,
        IUserRepository _userRepository,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<QMSDbContext> _logger,
        ICapaActionRepository _capaActionRepository,
        ICapaReviewRepository _capaReviewRepository) : IRequestHandler<UpdateCapaRequestCommand, ServiceResponse<CapaRequestDto>>
    {
        public async Task<ServiceResponse<CapaRequestDto>> Handle(UpdateCapaRequestCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var existEntity = await _capaRequestRepository.All
              .Where(c => c.Id == request.Id).FirstOrDefaultAsync();
                if (existEntity == null)
                {
                    return ServiceResponse<CapaRequestDto>.Return404("CapaRequest not  found");
                }
                var existTitle = await _capaRequestRepository.All
                    .Where(c => c.Id != request.Id && c.Title.ToLower() == request.Title.ToLower())
                    .FirstOrDefaultAsync();
                if (existTitle != null)
                {
                    return ServiceResponse<CapaRequestDto>.Return409("Title already exist");
                
                }
                if (request.CapaRequestStatus == CapaRequestStatus.CLOSED || request.CapaRequestStatus == CapaRequestStatus.REJECTED || request.CapaRequestStatus == CapaRequestStatus.REVIEWED)
                {
                    var actionsStatus = await _capaActionRepository.All
                        .Where(c => c.CapaRequestId == existEntity.Id
                        && (c.Status == CapaActionStatus.All ||
                            c.Status == CapaActionStatus.PENDING ||
                            c.Status == CapaActionStatus.INPROGRESS)).CountAsync();
                    var capaReviewStatus = await _capaReviewRepository.All.Where(
                       c => c.CapaRequestId == existEntity.Id && c.Status == ReviewStatus.NEEDS_REVISION).CountAsync();
                    if (actionsStatus >= 1 )
                    {
                        return ServiceResponse<CapaRequestDto>.Return409("All CAPA actions must be completed or rejected first.");
                    }
                    if (capaReviewStatus >= 1 )
                    {
                        return ServiceResponse<CapaRequestDto>.Return409("All CAPA reviews must be approved or rejected first.");
                    }
                }
                existEntity = _mapper.Map(request, existEntity);
                var capaRequestLog = _mapper.Map<CapaRequestLog>(existEntity);
                capaRequestLog.Id = Guid.NewGuid();
                capaRequestLog.LogStatus = LogStatus.Updated;
                _capaRequestRepository.Update(existEntity);
                _capaRequestLogRepository.Add(capaRequestLog);
                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<CapaRequestDto>.Return500();
                }

                var user = await _userRepository.All.Where(c => c.Id == request.RaisedById).FirstOrDefaultAsync();
                if (user != null)
                {
                    existEntity.RaisedBY = user;
                }
                var entityDto = _mapper.Map<CapaRequestDto>(existEntity);

                return ServiceResponse<CapaRequestDto>.ReturnResultWith200(entityDto);
            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Error while updating CapaRequest");
                return ServiceResponse<CapaRequestDto>.Return500("Error while updating CapaRequest");
            }


        }
    }
}
