﻿using Amazon.Runtime.Internal.Util;
using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class UpdateCapaReviewCommandHandler(
        ICapaReviewRepository _capaReviewRepository,
        IMapper _mapper,
        ILogger<UpdateCapaReviewCommandHandler> _logger,
        IUnitOfWork<QMSDbContext> _uow,
        ICapaActionRepository _capaActionRepository,
        ICapaRequestRepository _capaRequestRepository,
        UserInfoToken _userInfoToken) : IRequestHandler<UpdateCapaReviewCommand, ServiceResponse<CapaReviewDto>>
    {
        public async Task<ServiceResponse<CapaReviewDto>> Handle(UpdateCapaReviewCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var entityExist=await _capaReviewRepository.All.Where(c=>c.Id==request.Id).FirstOrDefaultAsync();
                if (entityExist == null)
                {
                    return ServiceResponse<CapaReviewDto>.Return404("capaReview not found");
                }
                var capaExist = await _capaRequestRepository.FindAsync(request.CapaRequestId);
                if (entityExist == null)
                {
                    return ServiceResponse<CapaReviewDto>.Return404("CapaRequest not found");
                }
                if (request.Status == ReviewStatus.APPROVED || request.Status == ReviewStatus.REJECTED)
                {
                    var actionsStatus = await _capaActionRepository.All
                        .Where(c => c.CapaRequestId == request.CapaRequestId
                        && (c.Status == CapaActionStatus.All ||
                            c.Status == CapaActionStatus.PENDING ||
                            c.Status == CapaActionStatus.INPROGRESS)).CountAsync();
                    if (actionsStatus >= 1)
                    {
                        return ServiceResponse<CapaReviewDto>.Return409("All CAPA actions must be completed or rejected first.");
                    }
                    capaExist.CapaRequestStatus = request.Status == ReviewStatus.APPROVED ? CapaRequestStatus.CLOSED : CapaRequestStatus.REJECTED;
                    _capaRequestRepository.Update(capaExist);
                }
                _mapper.Map(request, entityExist);
                entityExist.ReviewedById = _userInfoToken.Id;
                _capaReviewRepository.Update(entityExist);
                if (await _uow.SaveAsync() <= 0)
                {
                    return ServiceResponse<CapaReviewDto>.Return500("error while updating CapaReview ");
                }
                var dto = _mapper.Map<CapaReviewDto>(entityExist);
                return ServiceResponse<CapaReviewDto>.ReturnResultWith200(dto);
            }
            catch (Exception ex) 
            {
                _logger.LogError(ex, "An unexpected fault happened.");
                return ServiceResponse<CapaReviewDto>.Return500("An unexpected fault happened. Try again later.");
            }
        }
    }
}
