﻿using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;

namespace QualityManagement.MediatR;

public class AddCapaRootCauseCommandHandler(
    ICapaRootCauseRepository _capaRootCauseRepository,
    ICapaRequestRepository _capaRequestRepository,
    ICapaRootCauseMethodologyRepository _capaRootCauseMethodologyRepository,
    IUserRepository _userRepository,
    UserInfoToken userInfoToken,
    IMapper _mapper,
    IUnitOfWork<QMSDbContext> _uow,
    ILogger<AddCapaRootCauseCommandHandler> _logger) : IRequestHandler<AddCapaRootCauseCommand, ServiceResponse<CapaRootCauseDto>>
{
    public async Task<ServiceResponse<CapaRootCauseDto>> Handle(AddCapaRootCauseCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var entityExist = await _capaRequestRepository.FindAsync(request.CapaRequestId);
            if (entityExist == null)
            {
                return ServiceResponse<CapaRootCauseDto>.Return404("CapaRootCause not found");
            }
            var entity = _mapper.Map<CapaRootCause>(request);
            entity.CreatedDate = DateTime.UtcNow;
            entity.CreatedById = userInfoToken.Id;
            _capaRootCauseRepository.Add(entity);
            if (await _uow.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<CapaRootCauseDto>.Return500();
            }
            var capaRootCauseMethodology = await _capaRootCauseMethodologyRepository.All
                .Where(c => c.Id == request.MethodologyId).FirstOrDefaultAsync();
            if (capaRootCauseMethodology != null)
            {
                entity.CapaRootCauseMethodology = capaRootCauseMethodology;
            }
            var user = await _userRepository.All.Where(c => c.Id == entity.CreatedById).FirstOrDefaultAsync();
            if (user != null)
            {
                entity.CreatedBy = user;
            }
            var entityDto = _mapper.Map<CapaRootCauseDto>(entity);
            return ServiceResponse<CapaRootCauseDto>.ReturnResultWith201(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Error while saving CapaRootCause");
            return ServiceResponse<CapaRootCauseDto>.Return500("Error while saving CapaRootCause");
        }

    }
}
