﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR;

public class AddComplaintCommandHandler(
     IComplaintRepository _complaintRepository,
     IComplaintLogRepository _complaintLogRepository,
     UserInfoToken _userInfoToken,
    IMapper _mapper,
    IUnitOfWork<QMSDbContext> _uow,
    ILogger<AddComplaintCommandHandler> _logger,
     IUserRepository _userRepository,
    ISendEmailRepository _sendEmailRepository,
    IUserNotificationRepository _userNotificationRepository) : IRequestHandler<AddComplaintCommand, ServiceResponse<ComplaintDto>>
{
    public async Task<ServiceResponse<ComplaintDto>> Handle(AddComplaintCommand request, CancellationToken cancellationToken)
    {
        try
        {
            var entityExistTitle = await _complaintRepository.All
                .Where(c => c.Title.ToLower() == request.Title.ToLower())
                .FirstOrDefaultAsync();
            if (entityExistTitle != null)
            {
                return ServiceResponse<ComplaintDto>.Return409("Complaint title already exist");
            }

            var entity = _mapper.Map<Complaint>(request);
            entity.Id = Guid.NewGuid();
            var newComplaintNumber = await _complaintRepository.GenerateCompalintNumberAsync();
            entity.ComplaintNo = newComplaintNumber;

            var complaintLog = _mapper.Map<ComplaintLog>(entity);
            complaintLog.Id = Guid.NewGuid();
            complaintLog.ComplaintDate = DateTime.UtcNow;
            complaintLog.CreatedDate = DateTime.UtcNow;
            complaintLog.CreatedBy = _userInfoToken.Id;
            complaintLog.LogStatus = LogStatus.Created;

            _complaintRepository.Add(entity);
            _complaintLogRepository.Add(complaintLog);
            
            // Send Assignment Enmail
            try
            {
                var currentUserId = _userInfoToken.Id;

                var userIds = new[] { _userInfoToken.Id, entity.AssignedToId };
                var users = await _userRepository.All
                    .Where(u => userIds.Contains(u.Id))
                    .ToListAsync();

                var user = users.FirstOrDefault(u => u.Id == _userInfoToken.Id);
                var asignedUser = users.FirstOrDefault(u => u.Id == entity.AssignedToId);
                if (asignedUser != null)
                {
                    var mainTitle = "Complaint Assignment Notification";
                    var subject = "Complaint Assignment";
                    var emailMessage = $"You have been assigned a new complaint  <b>{entity.Title}</b>. Please log in to the system to review the complaint details, take the necessary actions, and update the status accordingly.";
                    _sendEmailRepository.AddAsignmentEmails(new SendEmail
                    {
                        Email = asignedUser.Email,
                        FromName = user.FirstName + ' ' + user.LastName,
                        ToName = asignedUser.FirstName + ' ' + asignedUser.LastName,
                        CreatedBy = user.Id,
                        CreatedDate = DateTime.UtcNow,
                    }, emailMessage, mainTitle, subject);

                    // Sent Notification
                    var notificationDto = new UserNotificationDto
                    {
                        Id = entity.Id,
                        UserId = entity.AssignedToId.Value,
                        NotificationsType = NotificationsType.COMPLAINT,
                        Title = entity.Title
                    };
                    _userNotificationRepository.AddAssigmentNotifcation(notificationDto);
                }
            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "email does not sent");
            }
            if (await _uow.SaveAsync(cancellationToken) < 0)
            {
                return ServiceResponse<ComplaintDto>.Return500();
            }
            var entityDto = _mapper.Map<ComplaintDto>(entity);
            return ServiceResponse<ComplaintDto>.ReturnResultWith201(entityDto);
        }
        catch (System.Exception ex)
        {
            _logger.LogError(ex, "Error while saving Complaint");
            return ServiceResponse<ComplaintDto>.Return500("Error while saving Complaint");
        }
    }
}
