﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class UpdateComplaintCommandHandler(
        IComplaintRepository _complaintRepository,
        IComplaintLogRepository _complaintLogRepository,
        IMapper _mapper,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<UpdateComplaintCommandHandler> _logger,
        IUserRepository _userRepository,
        IEmailSMTPSettingRepository _emailSMTPSettingRepository,
        ISendEmailRepository _sendEmailRepository,
        UserInfoToken _userInfoToken,
        IComplaintActionRepository _complaintActionRepository,
        IComplaintInvestigationRepository _complaintInvestigationRepository,
        IUserNotificationRepository _userNotificationRepository) : IRequestHandler<UpdateComplaintCommand, ServiceResponse<ComplaintDto>>
    {
        public async Task<ServiceResponse<ComplaintDto>> Handle(UpdateComplaintCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var entityExist = await _complaintRepository.All
                  .Where(c => c.Id == request.Id)
                  .FirstOrDefaultAsync();
                if (entityExist == null)
                {
                    return ServiceResponse<ComplaintDto>.Return404("Complaint not found");
                }

                var entityExistTitle = await _complaintRepository.All
                   .Where(c => c.Id != request.Id && c.Title.ToLower() == request.Title.ToLower())
                   .FirstOrDefaultAsync();
                if (entityExistTitle != null)
                {
                    return ServiceResponse<ComplaintDto>.Return409("Complaint title already exist");
                }

                if (request.Status == ComplaintStatusEnum.CLOSED || request.Status == ComplaintStatusEnum.RESOLVED)
                {
                    var complaintActionStatus = await _complaintActionRepository.All
                        .Where(c => c.ComplaintId == request.Id && (
                                    c.Status == ComplaintStatusEnum.OPEN ||
                                    c.Status == ComplaintStatusEnum.IN_PROGRESS ||
                                    c.Status == ComplaintStatusEnum.ESCALATED ||
                                    c.Status == ComplaintStatusEnum.PENDING)).CountAsync();
                    if (complaintActionStatus >= 1)
                    {
                        return ServiceResponse<ComplaintDto>.Return409("All Complaint actions must be closed or rejected first.");
                    }
                    var investigationStatus = await _complaintInvestigationRepository.All
                        .Where(c => c.ComplaintId == request.Id && (
                                c.Status == ComplaintStatusEnum.OPEN ||
                                c.Status == ComplaintStatusEnum.IN_PROGRESS ||
                                c.Status == ComplaintStatusEnum.ESCALATED ||
                                c.Status == ComplaintStatusEnum.PENDING)).CountAsync();
                    if (investigationStatus >= 1)
                    {
                        return ServiceResponse<ComplaintDto>.Return409("All Complaint investigation must be closed or rejected first.");
                    }
                }
                // Send Assignment Email and Notification If Assigned User Not are same 
                if (entityExist.AssignedToId != request.AssignedToId)
                {
                    var userIds = new[] { _userInfoToken.Id, request.AssignedToId };
                    var users = await _userRepository.All
                        .Where(u => userIds.Contains(u.Id))
                        .ToListAsync();

                    var user = users.FirstOrDefault(u => u.Id == _userInfoToken.Id);
                    var asignedUser = users.FirstOrDefault(u => u.Id == request.AssignedToId);
                    // Send Assignment Email
                    try
                    {
                        if (asignedUser != null)
                        {
                            var mainTitle = "Complaint Assignment Notification";
                            var subject = "Complaint Assignment";
                            var emailMessage = $"You have been assigned a new complaint : <b>{request.Title}</b>. Please log in to the system to review the complaint details, take the necessary actions, and update the status accordingly.";
                            _sendEmailRepository.AddAsignmentEmails(new SendEmail
                            {
                                Email = asignedUser.Email,
                                FromName = user.FirstName + ' ' + user.LastName,
                                ToName = asignedUser.FirstName + ' ' + asignedUser.LastName,
                                CreatedBy = user.Id,
                                CreatedDate = DateTime.UtcNow,
                            }, emailMessage, mainTitle, subject);

                            // Sent Notification
                            var notificationDto = new UserNotificationDto
                            {
                                Id = entityExist.Id,
                                UserId = request.AssignedToId,
                                NotificationsType = NotificationsType.COMPLAINT,
                                Title = request.Title
                            };
                            _userNotificationRepository.AddAssigmentNotifcation(notificationDto);
                        }
                    }
                    catch (System.Exception ex)
                    {
                        _logger.LogError(ex, "email does not sent");
                    }

                }
                var entity = _mapper.Map(request, entityExist);
                var complaintLog = _mapper.Map<ComplaintLog>(entity);
                complaintLog.Id = Guid.NewGuid();
                complaintLog.LogStatus = LogStatus.Updated;
                _complaintRepository.Update(entity);
                _complaintLogRepository.Add(complaintLog);

                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<ComplaintDto>.Return500();
                }
                entity = await _complaintRepository
                 .AllIncluding(c => c.AssignedTo, c => c.ComplaintType)
                 .Where(c => c.Id == request.Id)
                 .FirstOrDefaultAsync();
                var entityDto = _mapper.Map<ComplaintDto>(entity);
                return ServiceResponse<ComplaintDto>.ReturnResultWith201(entityDto);

            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Error while Updating Complaint");
                return ServiceResponse<ComplaintDto>.Return500("Error while Updating Complaint");
            }
        }
    }
}
