﻿using AutoMapper;
using MediatR;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Logging;
using QualityManagement.Common.UnitOfWork;
using QualityManagement.Data;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Entities;
using QualityManagement.Domain;
using QualityManagement.Helper;
using QualityManagement.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class AddComplaintActionCommandHanler(
        IComplaintActionRepository _complaintActionRepository,
        IMapper _mapper,
        IUserRepository _userRepository,
        IUnitOfWork<QMSDbContext> _uow,
        ILogger<AddComplaintActionCommandHanler> _logger,
        IComplaintInvestigationRepository _complaintInvestigationRepository,
        IComplaintRepository _complaintRepository,
        UserInfoToken _userInfoToken,
        ISendEmailRepository _sendEmailRepository,
        IUserNotificationRepository _userNotificationRepository) : IRequestHandler<AddComplaintActionCommand, ServiceResponse<ComplaintActionDto>>
    {
        public async Task<ServiceResponse<ComplaintActionDto>> Handle(AddComplaintActionCommand request, CancellationToken cancellationToken)
        {
            try
            {
                var complaint = await _complaintRepository.All.Where(c => c.Id == request.ComplaintId).FirstOrDefaultAsync();
                if (complaint == null)
                {
                    return ServiceResponse<ComplaintActionDto>.Return404("complaint not found");
                }
                var entity = _mapper.Map<ComplaintAction>(request);
                entity.ActionDate = DateTime.UtcNow;
                if (request.Status == ComplaintStatusEnum.RESOLVED || request.Status == ComplaintStatusEnum.CLOSED)
                {
                    var investigationStatus = await _complaintInvestigationRepository.All
                        .Where(c => c.ComplaintId == request.ComplaintId && (
                                    c.Status == ComplaintStatusEnum.OPEN ||
                                    c.Status == ComplaintStatusEnum.IN_PROGRESS ||
                                    c.Status == ComplaintStatusEnum.ESCALATED ||
                                    c.Status == ComplaintStatusEnum.PENDING)).CountAsync();
                    if (investigationStatus >= 1)
                    {
                        return ServiceResponse<ComplaintActionDto>.Return409("All Complaint investigation must be closed or rejected first.");
                    }
                }
                _complaintActionRepository.Add(entity);
                // Sent Email Responsible Person
                try
                {
                    var userIds = new[] { _userInfoToken.Id, entity.ResponsibleId };
                    var users = await _userRepository.All
                        .Where(u => userIds.Contains(u.Id))
                        .ToListAsync();

                    var fromUser = users.FirstOrDefault(u => u.Id == _userInfoToken.Id);
                    var responsibleUser = users.FirstOrDefault(u => u.Id == entity.ResponsibleId);
                    if (responsibleUser != null)
                    {
                        var mainTitle = "Complaint Action Assignment Notification";
                        var subject = "Complaint Action Assignment";
                        var emailMessage = $"You have been assigned as the Responsible Person for the complaint action: <b>{entity.ActionTitle}</b>.Please log in to the system to review the action details, perform the necessary steps, and ensure it is completed before the due date.";
                        _sendEmailRepository.AddAsignmentEmails(new SendEmail
                        {
                            Email = responsibleUser.Email,
                            FromName = fromUser.FirstName + ' ' + fromUser.LastName,
                            ToName = responsibleUser.FirstName + ' ' + responsibleUser.LastName,
                            CreatedBy = fromUser.Id,
                            CreatedDate = DateTime.UtcNow,
                        }, emailMessage, mainTitle, subject);

                        // Sent Notification
                        var notificationDto = new UserNotificationDto
                        {
                            Id = entity.Id,
                            UserId = entity.ResponsibleId,
                            NotificationsType = NotificationsType.COMPLAINT_ACTION,
                            Title = entity.ActionTitle
                        };
                        _userNotificationRepository.AddAssigmentNotifcation(notificationDto);
                    }
                }
                catch (System.Exception ex)
                {
                    _logger.LogError(ex, "email does not sent");
                }
                if (await _uow.SaveAsync(cancellationToken) < 0)
                {
                    return ServiceResponse<ComplaintActionDto>.Return500();
                }
                var user = await _userRepository.All.Where(c => c.Id == entity.ResponsibleId).FirstOrDefaultAsync();
                if (user != null)
                {
                    entity.Responsible = user;
                }
                var entityDto = _mapper.Map<ComplaintActionDto>(entity);
                return ServiceResponse<ComplaintActionDto>.ReturnResultWith201(entityDto);

            }
            catch (System.Exception ex)
            {
                _logger.LogError(ex, "Error while saving Complaint Action");
                return ServiceResponse<ComplaintActionDto>.Return500("Error while saving Complaint Action");
            }
        }
    }
}
