﻿using MediatR;
using Microsoft.EntityFrameworkCore;
using QualityManagement.Data.Dto;
using QualityManagement.Data.Dto.RiskManagement;
using QualityManagement.Helper;
using QualityManagement.Repository;
using QualityManagement.Repository.RiskManagment;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace QualityManagement.MediatR
{
    public class DownloadComplaintAttachmentCommandHandler(
          IComplaintAttchmentRepository _complaintAttchmentRepository,
         IComplaintAttachmentChunkRepository  _complaintAttachmentChunkRepository,
        IStorageSettingRepository _storageSettingRepository,
        StorageServiceFactory _storageServiceFactory) : IRequestHandler<DownloadComplaintAttachmentCommand, ServiceResponse<ComplaintAttachmentDownload>>

    {
        public async Task<ServiceResponse<ComplaintAttachmentDownload>> Handle(DownloadComplaintAttachmentCommand request, CancellationToken cancellationToken)
        {
            var complaintAttachment = _complaintAttchmentRepository.Find(request.ComplaintAttachmentId);
            if (complaintAttachment == null)
            {
                return ServiceResponse<ComplaintAttachmentDownload>.ReturnFailed(404, "Complaint Attachment is not found");
            }

            var complaintAttachmentChunk = await _complaintAttachmentChunkRepository.All
                .Where(c => c.ComplaintAttachmentId == request.ComplaintAttachmentId  && c.ChunkIndex == request.ChunkIndex)
                .FirstOrDefaultAsync();

            if (complaintAttachmentChunk == null)
            {
                return ServiceResponse<ComplaintAttachmentDownload>.ReturnFailed(404, "Complaint Attachment Chunk is not found");
            }

            var storeageSetting = await _storageSettingRepository.GetStorageSettingByIdOrLocal(complaintAttachment.StorageSettingId);

            if (storeageSetting == null)
            {
                return ServiceResponse<ComplaintAttachmentDownload>.ReturnFailed(404, "Storage setting not found");
            }
            var storageService = _storageServiceFactory.GetStorageService(storeageSetting.StorageType);
            var fileResult = await storageService.DownloadFileAsync(complaintAttachmentChunk.Url, storeageSetting.JsonValue, complaintAttachment.Key, complaintAttachment.IV);
            if (string.IsNullOrWhiteSpace(fileResult.ErrorMessage))
            {
                var extension = complaintAttachmentChunk.Extension;
                if (!extension.StartsWith("."))
                    extension = "." + extension;
                ComplaintAttachmentDownload complaintAttachmentDownload = new ComplaintAttachmentDownload
                {
                    Data = Convert.ToBase64String(fileResult.FileBytes),
                    ContentType = FileHelper.GetMimeTypeByExtension(extension),
                    FileName = complaintAttachmentChunk.Url,
                    ChunkIndex = complaintAttachmentChunk.ChunkIndex
                };
                return ServiceResponse<ComplaintAttachmentDownload>.ReturnResultWith200(complaintAttachmentDownload);
            }
            return ServiceResponse<ComplaintAttachmentDownload>.ReturnFailed(500, fileResult.ErrorMessage);

        }
    }
}
